const express = require('express')
const sqlite3 = require('sqlite3').verbose()
const cors = require('cors')
const { v4: uuidv4 } = require('uuid')

const app = express()
const port = 5000

app.use(cors())
app.use(express.json())

// Initialize database
const db = new sqlite3.Database('./database.sqlite')

db.serialize(() => {
  db.run(`
    CREATE TABLE IF NOT EXISTS posts (
      id TEXT PRIMARY KEY,
      title TEXT NOT NULL,
      content TEXT NOT NULL,
      topic TEXT NOT NULL,
      createdAt DATETIME DEFAULT CURRENT_TIMESTAMP,
      upvotes INTEGER DEFAULT 0
    )
  `)

  db.run(`
    CREATE TABLE IF NOT EXISTS comments (
      id TEXT PRIMARY KEY,
      postId TEXT NOT NULL,
      content TEXT NOT NULL,
      createdAt DATETIME DEFAULT CURRENT_TIMESTAMP,
      FOREIGN KEY (postId) REFERENCES posts (id)
    )
  `)

  db.run(`
    CREATE TABLE IF NOT EXISTS topics (
      id TEXT PRIMARY KEY,
      name TEXT NOT NULL UNIQUE,
      description TEXT
    )
  `)
})

// Routes
app.get('/api/posts', (req, res) => {
  const { topic, search } = req.query
  let query = 'SELECT * FROM posts'
  const params = []

  if (topic) {
    query += ' WHERE topic = ?'
    params.push(topic)
  }

  if (search) {
    query += topic ? ' AND' : ' WHERE'
    query += ' (title LIKE ? OR content LIKE ?)'
    params.push(`%${search}%`, `%${search}%`)
  }

  query += ' ORDER BY createdAt DESC'

  db.all(query, params, (err, rows) => {
    if (err) {
      return res.status(500).json({ error: err.message })
    }
    res.json(rows)
  })
})

app.post('/api/posts', (req, res) => {
  const { title, content, topic } = req.body
  const id = uuidv4()

  db.run(
    'INSERT INTO posts (id, title, content, topic) VALUES (?, ?, ?, ?)',
    [id, title, content, topic],
    function (err) {
      if (err) {
        return res.status(500).json({ error: err.message })
      }
      res.json({ id, title, content, topic })
    }
  )
})

app.get('/api/posts/:id', (req, res) => {
  const { id } = req.params

  db.get('SELECT * FROM posts WHERE id = ?', [id], (err, row) => {
    if (err) {
      return res.status(500).json({ error: err.message })
    }
    if (!row) {
      return res.status(404).json({ error: 'Post not found' })
    }
    res.json(row)
  })
})

app.post('/api/posts/:id/comments', (req, res) => {
  const { id } = req.params
  const { content } = req.body
  const commentId = uuidv4()

  db.run(
    'INSERT INTO comments (id, postId, content) VALUES (?, ?, ?)',
    [commentId, id, content],
    function (err) {
      if (err) {
        return res.status(500).json({ error: err.message })
      }
      res.json({ id: commentId, postId: id, content })
    }
  )
})

app.get('/api/posts/:id/comments', (req, res) => {
  const { id } = req.params

  db.all('SELECT * FROM comments WHERE postId = ? ORDER BY createdAt DESC', [id], (err, rows) => {
    if (err) {
      return res.status(500).json({ error: err.message })
    }
    res.json(rows)
  })
})

app.get('/api/topics', (req, res) => {
  db.all('SELECT * FROM topics ORDER BY name', [], (err, rows) => {
    if (err) {
      return res.status(500).json({ error: err.message })
    }
    res.json(rows)
  })
})

app.listen(port, () => {
  console.log(`Server running on port ${port}`)
})
