import { useEffect, useState } from 'react'
import { Box, Typography, Card, CardContent, Button, TextField, Select, MenuItem } from '@mui/material'
import { Link } from 'react-router-dom'

export default function Home() {
  const [posts, setPosts] = useState([])
  const [topics, setTopics] = useState([])
  const [selectedTopic, setSelectedTopic] = useState('')
  const [searchQuery, setSearchQuery] = useState('')

  useEffect(() => {
    fetchPosts()
    fetchTopics()
  }, [])

  const fetchPosts = () => {
    let url = '/api/posts'
    const params = new URLSearchParams()
    
    if (selectedTopic) params.append('topic', selectedTopic)
    if (searchQuery) params.append('search', searchQuery)
    
    if (params.toString()) url += `?${params.toString()}`

    fetch(url)
      .then(res => res.json())
      .then(data => setPosts(data))
      .catch(err => console.error(err))
  }

  const fetchTopics = () => {
    fetch('/api/topics')
      .then(res => res.json())
      .then(data => setTopics(data))
      .catch(err => console.error(err))
  }

  return (
    <Box sx={{ p: 3 }}>
      <Typography variant="h4" gutterBottom>
        Campus Discussions
      </Typography>
      
      <Box sx={{ display: 'flex', gap: 2, mb: 3 }}>
        <TextField
          label="Search posts"
          variant="outlined"
          value={searchQuery}
          onChange={(e) => setSearchQuery(e.target.value)}
          sx={{ flexGrow: 1 }}
        />
        <Button
          variant="contained"
          onClick={fetchPosts}
        >
          Search
        </Button>
      </Box>
      
      <Box sx={{ display: 'flex', gap: 2, mb: 3 }}>
        <Select
          value={selectedTopic}
          onChange={(e) => setSelectedTopic(e.target.value)}
          displayEmpty
          sx={{ minWidth: 200 }}
        >
          <MenuItem value="">All Topics</MenuItem>
          {topics.map(topic => (
            <MenuItem key={topic.id} value={topic.name}>
              {topic.name}
            </MenuItem>
          ))}
        </Select>
        <Button
          variant="contained"
          component={Link}
          to="/create"
        >
          New Post
        </Button>
      </Box>
      
      {posts.map(post => (
        <Card key={post.id} sx={{ mb: 2 }}>
          <CardContent>
            <Typography variant="h5" component={Link} to={`/post/${post.id}`}>
              {post.title}
            </Typography>
            <Typography variant="subtitle1" color="text.secondary">
              Topic: {post.topic}
            </Typography>
            <Typography variant="body2" sx={{ mt: 1 }}>
              {post.content.substring(0, 100)}...
            </Typography>
          </CardContent>
        </Card>
      ))}
    </Box>
  )
}
