import { useState, useEffect } from 'react'
import './App.css'

function App() {
  const [questions, setQuestions] = useState([])
  const [newQuestion, setNewQuestion] = useState('')
  const [answers, setAnswers] = useState({})

  useEffect(() => {
    fetchQuestions()
  }, [])

  const fetchQuestions = async () => {
    const response = await fetch('/api/questions')
    const data = await response.json()
    setQuestions(data)
    
    // Fetch answers for each question
    data.forEach(async (question) => {
      const answerResponse = await fetch(`/api/answers/${question.id}`)
      const answerData = await answerResponse.json()
      setAnswers(prev => ({...prev, [question.id]: answerData}))
    })
  }

  const submitQuestion = async () => {
    await fetch('/api/questions', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({ text: newQuestion }),
    })
    setNewQuestion('')
    fetchQuestions()
  }

  const submitAnswer = async (questionId, answerText) => {
    await fetch('/api/answers', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({ 
        questionId, 
        text: answerText 
      }),
    })
    fetchQuestions()
  }

  const rateAnswer = async (answerId, rating) => {
    await fetch('/api/answers/rate', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({ 
        answerId, 
        rating 
      }),
    })
    fetchQuestions()
  }

  return (
    <div className="app">
      <h1>Q&A Website</h1>
      
      <div className="question-form">
        <input
          type="text"
          value={newQuestion}
          onChange={(e) => setNewQuestion(e.target.value)}
          placeholder="Ask a question..."
        />
        <button onClick={submitQuestion}>Submit</button>
      </div>

      <div className="questions-list">
        {questions.map((question) => (
          <div key={question.id} className="question-card">
            <h3>{question.text}</h3>
            
            <div className="answers-section">
              {answers[question.id]?.map((answer) => (
                <div key={answer.id} className="answer-card">
                  <p>{answer.text}</p>
                  <div className="rating">
                    <span>Rating: {answer.rating || 0}</span>
                    <button onClick={() => rateAnswer(answer.id, 1)}>👍</button>
                    <button onClick={() => rateAnswer(answer.id, -1)}>👎</button>
                  </div>
                </div>
              ))}
              
              <div className="answer-form">
                <input
                  type="text"
                  id={`answer-${question.id}`}
                  placeholder="Your answer..."
                />
                <button onClick={() => {
                  const input = document.getElementById(`answer-${question.id}`)
                  submitAnswer(question.id, input.value)
                  input.value = ''
                }}>Answer</button>
              </div>
            </div>
          </div>
        ))}
      </div>
    </div>
  )
}

export default App
