import { useState, useEffect } from 'react'
import { FaSearch, FaBed, FaBath, FaRulerCombined } from 'react-icons/fa'

const App = () => {
  const [properties, setProperties] = useState([])
  const [filteredProperties, setFilteredProperties] = useState([])
  const [searchTerm, setSearchTerm] = useState('')
  const [filters, setFilters] = useState({
    minPrice: '',
    maxPrice: '',
    bedrooms: '',
    bathrooms: ''
  })

  useEffect(() => {
    // Mock data - in a real app, this would come from an API
    const mockProperties = [
      {
        id: 1,
        title: 'Modern Apartment',
        address: '123 Main St, City',
        price: 250000,
        bedrooms: 2,
        bathrooms: 1,
        sqft: 1200,
        description: 'Beautiful modern apartment with great views.',
        image: 'https://via.placeholder.com/400x300'
      },
      {
        id: 2,
        title: 'Cozy Cottage',
        address: '456 Oak Ave, Town',
        price: 180000,
        bedrooms: 3,
        bathrooms: 2,
        sqft: 1500,
        description: 'Charming cottage with a large backyard.',
        image: 'https://via.placeholder.com/400x300'
      },
      {
        id: 3,
        title: 'Luxury Villa',
        address: '789 Beach Blvd, Coastal',
        price: 750000,
        bedrooms: 4,
        bathrooms: 3,
        sqft: 3000,
        description: 'Stunning luxury villa with ocean views.',
        image: 'https://via.placeholder.com/400x300'
      }
    ]
    
    setProperties(mockProperties)
    setFilteredProperties(mockProperties)
  }, [])

  useEffect(() => {
    let results = properties
    
    // Apply search term filter
    if (searchTerm) {
      results = results.filter(property =>
        property.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
        property.address.toLowerCase().includes(searchTerm.toLowerCase()) ||
        property.description.toLowerCase().includes(searchTerm.toLowerCase())
      )
    }

    // Apply other filters
    if (filters.minPrice) {
      results = results.filter(property => property.price >= Number(filters.minPrice))
    }
    if (filters.maxPrice) {
      results = results.filter(property => property.price <= Number(filters.maxPrice))
    }
    if (filters.bedrooms) {
      results = results.filter(property => property.bedrooms >= Number(filters.bedrooms))
    }
    if (filters.bathrooms) {
      results = results.filter(property => property.bathrooms >= Number(filters.bathrooms))
    }

    setFilteredProperties(results)
  }, [searchTerm, filters, properties])

  const handleFilterChange = (e) => {
    const { name, value } = e.target
    setFilters(prev => ({
      ...prev,
      [name]: value
    }))
  }

  const [selectedProperty, setSelectedProperty] = useState(null)

  return (
    <div style={{ padding: '20px' }}>
      <h1 style={{ color: 'steelblue', marginBottom: '20px' }}>Real Estate Listings</h1>
      
      {!selectedProperty ? (
        <>
          <div style={{ marginBottom: '20px' }}>
            <div style={{ display: 'flex', alignItems: 'center', marginBottom: '10px' }}>
              <FaSearch style={{ marginRight: '10px', color: 'steelblue' }} />
              <input
                type="text"
                placeholder="Search properties..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                style={{ flex: 1, padding: '8px' }}
              />
            </div>
            
            <div style={{ display: 'flex', flexWrap: 'wrap', gap: '10px', marginBottom: '20px' }}>
              <div>
                <label>Min Price: </label>
                <input
                  type="number"
                  name="minPrice"
                  value={filters.minPrice}
                  onChange={handleFilterChange}
                  placeholder="$"
                />
              </div>
              <div>
                <label>Max Price: </label>
                <input
                  type="number"
                  name="maxPrice"
                  value={filters.maxPrice}
                  onChange={handleFilterChange}
                  placeholder="$"
                />
              </div>
              <div>
                <label>Bedrooms: </label>
                <input
                  type="number"
                  name="bedrooms"
                  value={filters.bedrooms}
                  onChange={handleFilterChange}
                  placeholder="Min"
                />
              </div>
              <div>
                <label>Bathrooms: </label>
                <input
                  type="number"
                  name="bathrooms"
                  value={filters.bathrooms}
                  onChange={handleFilterChange}
                  placeholder="Min"
                />
              </div>
              <button onClick={() => {
                setSearchTerm('')
                setFilters({
                  minPrice: '',
                  maxPrice: '',
                  bedrooms: '',
                  bathrooms: ''
                })
              }}>
                Clear Filters
              </button>
            </div>
          </div>
          
          <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fill, minmax(300px, 1fr))', gap: '20px' }}>
            {filteredProperties.map(property => (
              <div key={property.id} className="card" onClick={() => setSelectedProperty(property)}>
                <img 
                  src={property.image} 
                  alt={property.title} 
                  style={{ width: '100%', height: '200px', objectFit: 'cover', borderRadius: '4px' }}
                />
                <h3 style={{ margin: '10px 0' }}>{property.title}</h3>
                <p>{property.address}</p>
                <p style={{ fontWeight: 'bold', margin: '10px 0' }}>${property.price.toLocaleString()}</p>
                <div style={{ display: 'flex', justifyContent: 'space-between' }}>
                  <span><FaBed /> {property.bedrooms} beds</span>
                  <span><FaBath /> {property.bathrooms} baths</span>
                  <span><FaRulerCombined /> {property.sqft} sqft</span>
                </div>
              </div>
            ))}
          </div>
        </>
      ) : (
        <div className="card" style={{ maxWidth: '800px', margin: '0 auto' }}>
          <button 
            onClick={() => setSelectedProperty(null)}
            style={{ marginBottom: '20px' }}
          >
            Back to Listings
          </button>
          <img 
            src={selectedProperty.image} 
            alt={selectedProperty.title} 
            style={{ width: '100%', height: '400px', objectFit: 'cover', borderRadius: '4px' }}
          />
          <h2 style={{ margin: '20px 0' }}>{selectedProperty.title}</h2>
          <p style={{ fontSize: '1.2rem', marginBottom: '10px' }}>{selectedProperty.address}</p>
          <p style={{ fontSize: '1.5rem', fontWeight: 'bold', margin: '20px 0' }}>
            ${selectedProperty.price.toLocaleString()}
          </p>
          <div style={{ display: 'flex', gap: '20px', marginBottom: '20px' }}>
            <span><FaBed /> {selectedProperty.bedrooms} Bedrooms</span>
            <span><FaBath /> {selectedProperty.bathrooms} Bathrooms</span>
            <span><FaRulerCombined /> {selectedProperty.sqft} Square Feet</span>
          </div>
          <p style={{ lineHeight: '1.6' }}>{selectedProperty.description}</p>
        </div>
      )}
    </div>
  )
}

export default App
