const express = require('express');
const sqlite3 = require('sqlite3').verbose();
const bodyParser = require('body-parser');
const cors = require('cors');
const jwt = require('jsonwebtoken');
const bcrypt = require('bcryptjs');
const Web3 = require('web3');
require('dotenv').config();

const app = express();
const PORT = process.env.PORT || 5000;
const web3 = new Web3(process.env.ETHEREUM_NODE || 'https://mainnet.infura.io/v3/YOUR_INFURA_KEY');

app.use(cors());
app.use(bodyParser.json());

const db = new sqlite3.Database(process.env.DATABASE_URL, (err) => {
  if (err) {
    console.error(err.message);
  }
  console.log('Connected to the escrow database.');
});

// Initialize database tables
db.serialize(() => {
  db.run(`
    CREATE TABLE IF NOT EXISTS users (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      username TEXT UNIQUE NOT NULL,
      password TEXT NOT NULL,
      wallet_address TEXT UNIQUE NOT NULL,
      role TEXT NOT NULL CHECK(role IN ('buyer', 'seller', 'admin'))
    )
  `);

  db.run(`
    CREATE TABLE IF NOT EXISTS products (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      seller_id INTEGER NOT NULL,
      name TEXT NOT NULL,
      description TEXT,
      price REAL NOT NULL,
      cryptocurrency TEXT NOT NULL,
      created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
      FOREIGN KEY (seller_id) REFERENCES users (id)
    )
  `);

  db.run(`
    CREATE TABLE IF NOT EXISTS transactions (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      product_id INTEGER NOT NULL,
      buyer_id INTEGER NOT NULL,
      seller_id INTEGER NOT NULL,
      amount REAL NOT NULL,
      escrow_address TEXT NOT NULL,
      status TEXT NOT NULL CHECK(status IN ('pending', 'confirmed', 'completed', 'disputed', 'cancelled')),
      created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
      updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
      FOREIGN KEY (product_id) REFERENCES products (id),
      FOREIGN KEY (buyer_id) REFERENCES users (id),
      FOREIGN KEY (seller_id) REFERENCES users (id)
    )
  `);

  db.run(`
    CREATE TABLE IF NOT EXISTS disputes (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      transaction_id INTEGER NOT NULL,
      initiator_id INTEGER NOT NULL,
      reason TEXT NOT NULL,
      status TEXT NOT NULL CHECK(status IN ('open', 'resolved', 'rejected')),
      created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
      FOREIGN KEY (transaction_id) REFERENCES transactions (id),
      FOREIGN KEY (initiator_id) REFERENCES users (id)
    )
  `);
});

// Auth middleware
const authenticateJWT = (req, res, next) => {
  const authHeader = req.headers.authorization;
  if (authHeader) {
    const token = authHeader.split(' ')[1];
    jwt.verify(token, process.env.JWT_SECRET, (err, user) => {
      if (err) {
        return res.sendStatus(403);
      }
      req.user = user;
      next();
    });
  } else {
    res.sendStatus(401);
  }
};

const authorizeRole = (roles) => {
  return (req, res, next) => {
    if (!roles.includes(req.user.role)) {
      return res.sendStatus(403);
    }
    next();
  };
};

// Utility function to generate escrow address
const generateEscrowAddress = () => {
  return web3.eth.accounts.create().address;
};

// Auth Routes
app.post('/api/auth/register', async (req, res) => {
  const { username, password, wallet_address, role } = req.body;
  
  if (!web3.utils.isAddress(wallet_address)) {
    return res.status(400).json({ error: 'Invalid wallet address' });
  }

  const hashedPassword = await bcrypt.hash(password, 10);
  
  db.run(
    'INSERT INTO users (username, password, wallet_address, role) VALUES (?, ?, ?, ?)',
    [username, hashedPassword, wallet_address, role],
    function(err) {
      if (err) {
        return res.status(400).json({ error: err.message });
      }
      res.status(201).json({ id: this.lastID });
    }
  );
});

app.post('/api/auth/login', (req, res) => {
  const { username, password } = req.body;
  
  db.get('SELECT * FROM users WHERE username = ?', [username], async (err, user) => {
    if (err || !user || !(await bcrypt.compare(password, user.password))) {
      return res.status(401).json({ error: 'Invalid credentials' });
    }
    
    const token = jwt.sign(
      { id: user.id, username: user.username, role: user.role, wallet_address: user.wallet_address },
      process.env.JWT_SECRET,
      { expiresIn: '1h' }
    );
    
    res.json({ token, user: { id: user.id, username: user.username, role: user.role, wallet_address: user.wallet_address } });
  });
});

// Product Routes
app.get('/api/products', (req, res) => {
  const query = req.query.seller_id 
    ? 'SELECT * FROM products WHERE seller_id = ?'
    : 'SELECT * FROM products';
  
  const params = req.query.seller_id ? [req.query.seller_id] : [];

  db.all(query, params, (err, products) => {
    if (err) {
      return res.status(500).json({ error: err.message });
    }
    res.json(products);
  });
});

app.post('/api/products', authenticateJWT, authorizeRole(['seller', 'admin']), (req, res) => {
  const { name, description, price, cryptocurrency } = req.body;
  
  db.run(
    'INSERT INTO products (seller_id, name, description, price, cryptocurrency) VALUES (?, ?, ?, ?, ?)',
    [req.user.id, name, description, price, cryptocurrency],
    function(err) {
      if (err) {
        return res.status(400).json({ error: err.message });
      }
      res.status(201).json({ id: this.lastID });
    }
  );
});

// Transaction Routes
app.post('/api/transactions', authenticateJWT, async (req, res) => {
  const { product_id } = req.body;
  
  try {
    // Get product details
    const product = await new Promise((resolve, reject) => {
      db.get('SELECT * FROM products WHERE id = ?', [product_id], (err, row) => {
        if (err) reject(err);
        resolve(row);
      });
    });

    if (!product) {
      return res.status(404).json({ error: 'Product not found' });
    }

    // Generate escrow address
    const escrow_address = generateEscrowAddress();
    
    db.run(
      `INSERT INTO transactions 
      (product_id, buyer_id, seller_id, amount, escrow_address, status) 
      VALUES (?, ?, ?, ?, ?, ?)`,
      [product_id, req.user.id, product.seller_id, product.price, escrow_address, 'pending'],
      function(err) {
        if (err) {
          return res.status(400).json({ error: err.message });
        }
        
        // Return transaction with product details
        const transaction = {
          id: this.lastID,
          product_id,
          buyer_id: req.user.id,
          seller_id: product.seller_id,
          amount: product.price,
          escrow_address,
          status: 'pending',
          product_details: product
        };
        
        res.status(201).json(transaction);
      }
    );
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
});

app.put('/api/transactions/:id/confirm', authenticateJWT, (req, res) => {
  const { id } = req.params;
  
  db.run(
    `UPDATE transactions SET status = 'confirmed', updated_at = CURRENT_TIMESTAMP 
     WHERE id = ? AND buyer_id = ? AND status = 'pending'`,
    [id, req.user.id],
    function(err) {
      if (err) {
        return res.status(400).json({ error: err.message });
      }
      if (this.changes === 0) {
        return res.status(404).json({ error: 'Transaction not found or not eligible for confirmation' });
      }
      res.json({ message: 'Transaction confirmed' });
    }
  );
});

app.put('/api/transactions/:id/complete', authenticateJWT, (req, res) => {
  const { id } = req.params;
  
  db.run(
    `UPDATE transactions SET status = 'completed', updated_at = CURRENT_TIMESTAMP 
     WHERE id = ? AND seller_id = ? AND status = 'confirmed'`,
    [id, req.user.id],
    function(err) {
      if (err) {
        return res.status(400).json({ error: err.message });
      }
      if (this.changes === 0) {
        return res.status(404).json({ error: 'Transaction not found or not eligible for completion' });
      }
      res.json({ message: 'Transaction completed' });
    }
  );
});

// Dispute Routes
app.post('/api/disputes', authenticateJWT, (req, res) => {
  const { transaction_id, reason } = req.body;
  
  db.run(
    `INSERT INTO disputes (transaction_id, initiator_id, reason, status)
     VALUES (?, ?, ?, 'open')`,
    [transaction_id, req.user.id, reason],
    function(err) {
      if (err) {
        return res.status(400).json({ error: err.message });
      }
      
      // Update transaction status to disputed
      db.run(
        `UPDATE transactions SET status = 'disputed', updated_at = CURRENT_TIMESTAMP
         WHERE id = ?`,
        [transaction_id]
      );
      
      res.status(201).json({ id: this.lastID });
    }
  );
});

// Admin Routes
app.get('/api/admin/transactions', authenticateJWT, authorizeRole(['admin']), (req, res) => {
  db.all('SELECT * FROM transactions', [], (err, transactions) => {
    if (err) {
      return res.status(500).json({ error: err.message });
    }
    res.json(transactions);
  });
});

app.get('/api/admin/disputes', authenticateJWT, authorizeRole(['admin']), (req, res) => {
  db.all('SELECT * FROM disputes WHERE status = "open"', [], (err, disputes) => {
    if (err) {
      return res.status(500).json({ error: err.message });
    }
    res.json(disputes);
  });
});

app.put('/api/admin/disputes/:id/resolve', authenticateJWT, authorizeRole(['admin']), (req, res) => {
  const { id } = req.params;
  const { decision } = req.body; // 'refund' or 'release'
  
  // Get dispute details
  db.get('SELECT * FROM disputes WHERE id = ?', [id], (err, dispute) => {
    if (err || !dispute) {
      return res.status(404).json({ error: 'Dispute not found' });
    }
    
    // Update dispute status
    db.run(
      `UPDATE disputes SET status = ? WHERE id = ?`,
      ['resolved', id],
      function(err) {
        if (err) {
          return res.status(400).json({ error: err.message });
        }
        
        // Update transaction status based on decision
        const newStatus = decision === 'refund' ? 'cancelled' : 'completed';
        db.run(
          `UPDATE transactions SET status = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?`,
          [newStatus, dispute.transaction_id]
        );
        
        res.json({ message: `Dispute resolved with decision: ${decision}` });
      }
    );
  });
});

// Start server
app.listen(PORT, () => {
  console.log(`Server running on port ${PORT}`);
});
