import { useState } from 'react'
import { format } from 'date-fns'
import { FaBed, FaBath } from 'react-icons/fa'

export default function ApartmentCard({ apartment }) {
  const [showBookingForm, setShowBookingForm] = useState(false)
  const [bookingDates, setBookingDates] = useState({
    checkIn: '',
    checkOut: ''
  })

  const handleBookingChange = (e) => {
    const { name, value } = e.target
    setBookingDates(prev => ({ ...prev, [name]: value }))
  }

  const handleBook = (e) => {
    e.preventDefault()
    alert(`Booking confirmed for ${apartment.title} from ${bookingDates.checkIn} to ${bookingDates.checkOut}`)
    setShowBookingForm(false)
  }

  return (
    <div className="bg-white rounded-lg shadow-md overflow-hidden hover:shadow-lg transition-shadow">
      <img 
        src={apartment.image} 
        alt={apartment.title}
        className="w-full h-48 object-cover"
      />
      
      <div className="p-4">
        <h3 className="text-xl font-semibold mb-1">{apartment.title}</h3>
        <p className="text-gray-600 mb-2">{apartment.location}</p>
        
        <div className="flex justify-between items-center mb-4">
          <div>
            <span className="text-primary font-bold text-lg">${apartment.price}</span>
            <span className="text-gray-500"> / night</span>
          </div>
          <div className="flex space-x-2 text-sm items-center">
            <span className="flex items-center">
              <FaBed className="mr-1" /> {apartment.bedrooms}
            </span>
            <span>•</span>
            <span className="flex items-center">
              <FaBath className="mr-1" /> {apartment.bathrooms}
            </span>
          </div>
        </div>
        
        <div className="text-sm text-gray-500 mb-4">
          Available: {format(new Date(apartment.availableFrom), 'MMM d, yyyy')} - {format(new Date(apartment.availableTo), 'MMM d, yyyy')}
        </div>
        
        {!showBookingForm ? (
          <button
            onClick={() => setShowBookingForm(true)}
            className="w-full py-2 bg-primary text-white rounded hover:bg-opacity-90 transition"
          >
            Book Now
          </button>
        ) : (
          <form onSubmit={handleBook} className="space-y-3">
            <div>
              <label className="block text-sm mb-1">Check-in</label>
              <input
                type="date"
                name="checkIn"
                value={bookingDates.checkIn}
                onChange={handleBookingChange}
                min={apartment.availableFrom}
                max={apartment.availableTo}
                className="w-full p-2 border rounded text-sm focus:ring-primary"
                required
              />
            </div>
            
            <div>
              <label className="block text-sm mb-1">Check-out</label>
              <input
                type="date"
                name="checkOut"
                value={bookingDates.checkOut}
                onChange={handleBookingChange}
                min={apartment.availableFrom}
                max={apartment.availableTo}
                className="w-full p-2 border rounded text-sm focus:ring-primary"
                required
              />
            </div>
            
            <div className="flex space-x-2">
              <button
                type="button"
                onClick={() => setShowBookingForm(false)}
                className="flex-1 py-1.5 bg-gray-200 text-gray-800 rounded"
              >
                Cancel
              </button>
              <button
                type="submit"
                className="flex-1 py-1.5 bg-primary text-white rounded hover:bg-opacity-90"
              >
                Confirm
              </button>
            </div>
          </form>
        )}
      </div>
    </div>
  )
}
