const express = require('express');
const sqlite3 = require('sqlite3').verbose();
const bodyParser = require('body-parser');
const cors = require('cors');

const app = express();
const PORT = 5000;

app.use(cors());
app.use(bodyParser.json());

const db = new sqlite3.Database('./manufacturing.db');

// Initialize database tables
db.serialize(() => {
    db.run(`CREATE TABLE IF NOT EXISTS raw_materials (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        name TEXT NOT NULL,
        quantity REAL NOT NULL,
        unit TEXT NOT NULL,
        cost_per_unit REAL NOT NULL,
        supplier TEXT,
        last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    )`);

    db.run(`CREATE TABLE IF NOT EXISTS finished_products (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        name TEXT NOT NULL,
        size TEXT NOT NULL,
        quantity INTEGER NOT NULL,
        cost_per_unit REAL NOT NULL,
        selling_price REAL NOT NULL,
        last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    )`);

    db.run(`CREATE TABLE IF NOT EXISTS production (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        product_id INTEGER NOT NULL,
        quantity INTEGER NOT NULL,
        date TEXT NOT NULL,
        FOREIGN KEY (product_id) REFERENCES finished_products(id)
    )`);

    db.run(`CREATE TABLE IF NOT EXISTS customers (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        name TEXT NOT NULL,
        email TEXT,
        phone TEXT,
        address TEXT,
        company TEXT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    )`);

    db.run(`CREATE TABLE IF NOT EXISTS sales (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        customer_id INTEGER,
        product_id INTEGER NOT NULL,
        quantity INTEGER NOT NULL,
        total_price REAL NOT NULL,
        date TEXT NOT NULL,
        FOREIGN KEY (customer_id) REFERENCES customers(id),
        FOREIGN KEY (product_id) REFERENCES finished_products(id)
    )`);
});

// Raw Materials Routes
app.get('/api/raw-materials', (req, res) => {
    db.all('SELECT * FROM raw_materials', [], (err, rows) => {
        if (err) {
            res.status(400).json({ error: err.message });
            return;
        }
        res.json(rows);
    });
});

app.post('/api/raw-materials', (req, res) => {
    const { name, quantity, unit, cost_per_unit, supplier } = req.body;
    db.run(
        'INSERT INTO raw_materials (name, quantity, unit, cost_per_unit, supplier) VALUES (?, ?, ?, ?, ?)',
        [name, quantity, unit, cost_per_unit, supplier],
        function (err) {
            if (err) {
                res.status(400).json({ error: err.message });
                return;
            }
            res.json({ id: this.lastID });
        }
    );
});

// Finished Products Routes
app.get('/api/finished-products', (req, res) => {
    db.all('SELECT * FROM finished_products', [], (err, rows) => {
        if (err) {
            res.status(400).json({ error: err.message });
            return;
        }
        res.json(rows);
    });
});

app.post('/api/finished-products', (req, res) => {
    const { name, size, quantity, cost_per_unit, selling_price } = req.body;
    db.run(
        'INSERT INTO finished_products (name, size, quantity, cost_per_unit, selling_price) VALUES (?, ?, ?, ?, ?)',
        [name, size, quantity, cost_per_unit, selling_price],
        function (err) {
            if (err) {
                res.status(400).json({ error: err.message });
                return;
            }
            res.json({ id: this.lastID });
        }
    );
});

// Production Routes
app.get('/api/production', (req, res) => {
    db.all('SELECT * FROM production', [], (err, rows) => {
        if (err) {
            res.status(400).json({ error: err.message });
            return;
        }
        res.json(rows);
    });
});

app.post('/api/production', (req, res) => {
    const { product_id, quantity, date } = req.body;
    db.run(
        'INSERT INTO production (product_id, quantity, date) VALUES (?, ?, ?)',
        [product_id, quantity, date],
        function (err) {
            if (err) {
                res.status(400).json({ error: err.message });
                return;
            }
            res.json({ id: this.lastID });
        }
    );
});

// Customers Routes
app.get('/api/customers', (req, res) => {
    db.all('SELECT * FROM customers', [], (err, rows) => {
        if (err) {
            res.status(400).json({ error: err.message });
            return;
        }
        res.json(rows);
    });
});

app.post('/api/customers', (req, res) => {
    const { name, email, phone, address, company } = req.body;
    db.run(
        'INSERT INTO customers (name, email, phone, address, company) VALUES (?, ?, ?, ?, ?)',
        [name, email, phone, address, company],
        function (err) {
            if (err) {
                res.status(400).json({ error: err.message });
                return;
            }
            res.json({ id: this.lastID });
        }
    );
});

// Sales Routes
app.get('/api/sales', (req, res) => {
    db.all('SELECT * FROM sales', [], (err, rows) => {
        if (err) {
            res.status(400).json({ error: err.message });
            return;
        }
        res.json(rows);
    });
});

app.post('/api/sales', (req, res) => {
    const { customer_id, product_id, quantity, total_price, date } = req.body;
    db.run(
        'INSERT INTO sales (customer_id, product_id, quantity, total_price, date) VALUES (?, ?, ?, ?, ?)',
        [customer_id, product_id, quantity, total_price, date],
        function (err) {
            if (err) {
                res.status(400).json({ error: err.message });
                return;
            }
            res.json({ id: this.lastID });
        }
    );
});

// Reports
app.get('/api/reports/inventory', (req, res) => {
    db.all(`
        SELECT name, quantity, unit 
        FROM raw_materials
        UNION ALL
        SELECT name, quantity, 'pcs' as unit
        FROM finished_products
    `, [], (err, rows) => {
        if (err) {
            res.status(400).json({ error: err.message });
            return;
        }
        res.json(rows);
    });
});

app.get('/api/reports/sales', (req, res) => {
    db.all(`
        SELECT s.date, c.name as customer, p.name as product, s.quantity, s.total_price
        FROM sales s
        LEFT JOIN customers c ON s.customer_id = c.id
        LEFT JOIN finished_products p ON s.product_id = p.id
        ORDER BY s.date DESC
    `, [], (err, rows) => {
        if (err) {
            res.status(400).json({ error: err.message });
            return;
        }
        res.json(rows);
    });
});

app.listen(PORT, () => {
    console.log(`Server running on port ${PORT}`);
});
