import { useState, useEffect } from 'react'
import { useNavigate } from 'react-router-dom'

export default function Dashboard({ user, onLogout }) {
  const [attendance, setAttendance] = useState([])
  const [clockedIn, setClockedIn] = useState(false)
  const [message, setMessage] = useState('')
  const navigate = useNavigate()

  useEffect(() => {
    fetchAttendance()
    checkClockStatus()
  }, [])

  const fetchAttendance = async () => {
    try {
      const response = await fetch('http://localhost:3001/api/my-attendance', {
        headers: { 'Authorization': user.id }
      })
      const data = await response.json()
      setAttendance(data)
    } catch (err) {
      console.error(err)
    }
  }

  const checkClockStatus = async () => {
    const today = new Date().toISOString().split('T')[0]
    const response = await fetch('http://localhost:3001/api/my-attendance', {
      headers: { 'Authorization': user.id }
    })
    const data = await response.json()
    const todayRecord = data.find(record => record.date === today)
    setClockedIn(todayRecord?.clock_in && !todayRecord?.clock_out)
  }

  const handleClockIn = async () => {
    try {
      const response = await fetch('http://localhost:3001/api/clock-in', {
        method: 'POST',
        headers: { 
          'Content-Type': 'application/json',
          'Authorization': user.id
        }
      })
      const data = await response.json()
      setMessage(data.message)
      setClockedIn(true)
      fetchAttendance()
    } catch (err) {
      console.error(err)
    }
  }

  const handleClockOut = async () => {
    try {
      const response = await fetch('http://localhost:3001/api/clock-out', {
        method: 'POST',
        headers: { 
          'Content-Type': 'application/json',
          'Authorization': user.id
        }
      })
      const data = await response.json()
      setMessage(data.message)
      setClockedIn(false)
      fetchAttendance()
    } catch (err) {
      console.error(err)
    }
  }

  const handleLogout = () => {
    onLogout()
    navigate('/')
  }

  return (
    <div className="dashboard">
      <header>
        <h2>Welcome, {user.name}</h2>
        <button onClick={handleLogout}>Logout</button>
      </header>
      <div className="clock-actions">
        {!clockedIn ? (
          <button onClick={handleClockIn}>Clock In</button>
        ) : (
          <button onClick={handleClockOut}>Clock Out</button>
        )}
        {message && <p>{message}</p>}
      </div>
      <div className="attendance-history">
        <h3>Your Attendance Records</h3>
        <table>
          <thead>
            <tr>
              <th>Date</th>
              <th>Clock In</th>
              <th>Clock Out</th>
              <th>Status</th>
            </tr>
          </thead>
          <tbody>
            {attendance.map((record, index) => (
              <tr key={index}>
                <td>{record.date}</td>
                <td>{record.clock_in || '-'}</td>
                <td>{record.clock_out || '-'}</td>
                <td>{record.status}</td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>
    </div>
  )
}
