const express = require('express')
const cors = require('cors')
const bodyParser = require('body-parser')
const sqlite3 = require('sqlite3').verbose()
const bcrypt = require('bcryptjs')
const jwt = require('jsonwebtoken')
const app = express()

app.use(cors())
app.use(bodyParser.json())

const db = new sqlite3.Database('./internship.db', (err) => {
  if (err) {
    console.error(err.message)
  }
  console.log('Connected to the internship database.')
})

// Initialize database
db.serialize(() => {
  db.run(`
    CREATE TABLE IF NOT EXISTS users (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      email TEXT UNIQUE,
      password TEXT,
      type TEXT CHECK(type IN ('student', 'company')),
      name TEXT
    )
  `)

  db.run(`
    CREATE TABLE IF NOT EXISTS internships (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      company_id INTEGER,
      title TEXT,
      description TEXT,
      requirements TEXT,
      duration TEXT,
      FOREIGN KEY (company_id) REFERENCES users(id)
    )
  `)

  db.run(`
    CREATE TABLE IF NOT EXISTS applications (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      student_id INTEGER,
      internship_id INTEGER,
      status TEXT DEFAULT 'pending',
      application_date TEXT DEFAULT CURRENT_TIMESTAMP,
      FOREIGN KEY (student_id) REFERENCES users(id),
      FOREIGN KEY (internship_id) REFERENCES internships(id)
    )
  `)
})

// JWT Secret
const JWT_SECRET = 'your_jwt_secret_here' // In production, use environment variable

// Authentication middleware
const authenticateToken = (req, res, next) => {
  const authHeader = req.headers['authorization']
  const token = authHeader && authHeader.split(' ')[1]
  
  if (!token) return res.sendStatus(401)

  jwt.verify(token, JWT_SECRET, (err, user) => {
    if (err) return res.sendStatus(403)
    req.user = user
    next()
  })
}

// Routes
app.post('/api/register', async (req, res) => {
  const { email, password, type, name } = req.body
  
  try {
    const hashedPassword = await bcrypt.hash(password, 10)
    db.run(
      'INSERT INTO users (email, password, type, name) VALUES (?, ?, ?, ?)',
      [email, hashedPassword, type, name],
      function(err) {
        if (err) {
          return res.status(400).json({ error: 'Email already exists' })
        }
        res.status(201).json({ id: this.lastID })
      }
    )
  } catch (err) {
    res.status(500).json({ error: 'Registration failed' })
  }
})

app.post('/api/login', (req, res) => {
  const { email, password } = req.body
  
  db.get('SELECT * FROM users WHERE email = ?', [email], async (err, user) => {
    if (err || !user) {
      return res.status(401).json({ error: 'Invalid credentials' })
    }
    
    try {
      const validPassword = await bcrypt.compare(password, user.password)
      if (!validPassword) {
        return res.status(401).json({ error: 'Invalid credentials' })
      }
      
      const token = jwt.sign(
        { id: user.id, email: user.email, type: user.type },
        JWT_SECRET,
        { expiresIn: '1h' }
      )
      
      res.json({ token, userType: user.type })
    } catch (err) {
      res.status(500).json({ error: 'Login failed' })
    }
  })
})

// Internship routes
app.get('/api/internships', (req, res) => {
  db.all(`
    SELECT i.*, u.name as company_name 
    FROM internships i
    JOIN users u ON i.company_id = u.id
  `, [], (err, internships) => {
    if (err) {
      return res.status(500).json({ error: 'Failed to fetch internships' })
    }
    res.json(internships)
  })
})

app.post('/api/internships', authenticateToken, (req, res) => {
  if (req.user.type !== 'company') {
    return res.sendStatus(403)
  }
  
  const { title, description, requirements, duration } = req.body
  db.run(
    'INSERT INTO internships (company_id, title, description, requirements, duration) VALUES (?, ?, ?, ?, ?)',
    [req.user.id, title, description, requirements, duration],
    function(err) {
      if (err) {
        return res.status(400).json({ error: 'Failed to create internship' })
      }
      res.status(201).json({ id: this.lastID })
    }
  )
})

// Application routes
app.post('/api/applications', authenticateToken, (req, res) => {
  if (req.user.type !== 'student') {
    return res.sendStatus(403)
  }
  
  const { internship_id } = req.body
  db.run(
    'INSERT INTO applications (student_id, internship_id) VALUES (?, ?)',
    [req.user.id, internship_id],
    function(err) {
      if (err) {
        return res.status(400).json({ error: 'Failed to submit application' })
      }
      res.status(201).json({ id: this.lastID })
    }
  )
})

app.get('/api/applications', authenticateToken, (req, res) => {
  if (req.user.type === 'student') {
    db.all(`
      SELECT a.*, i.title as internship_title, u.name as company_name
      FROM applications a
      JOIN internships i ON a.internship_id = i.id
      JOIN users u ON i.company_id = u.id
      WHERE a.student_id = ?
    `, [req.user.id], (err, applications) => {
      if (err) {
        return res.status(500).json({ error: 'Failed to fetch applications' })
      }
      res.json(applications)
    })
  } else if (req.user.type === 'company') {
    db.all(`
      SELECT a.*, i.title as internship_title, u.name as student_name
      FROM applications a
      JOIN internships i ON a.internship_id = i.id
      JOIN users u ON a.student_id = u.id
      WHERE i.company_id = ?
    `, [req.user.id], (err, applications) => {
      if (err) {
        return res.status(500).json({ error: 'Failed to fetch applications' })
      }
      res.json(applications)
    })
  }
})

// Error handling middleware
app.use((err, req, res, next) => {
  console.error(err.stack)
  res.status(500).json({ error: 'Something went wrong!' })
})

const PORT = 5000
app.listen(PORT, () => {
  console.log(`Server running on port ${PORT}`)
})
