import { useState, useEffect } from 'react'
import ReactConfetti from 'react-confetti'
import './App.css'

const wasteItems = [
  { id: 1, name: 'Plastic Bottle', category: 'recycling', image: '♻️' },
  { id: 2, name: 'Banana Peel', category: 'compost', image: '🍌' },
  { id: 3, name: 'Newspaper', category: 'recycling', image: '📰' },
  { id: 4, name: 'Battery', category: 'hazardous', image: '🔋' },
  { id: 5, name: 'Apple Core', category: 'compost', image: '🍎' },
  { id: 6, name: 'Glass Jar', category: 'recycling', image: '🥫' },
  { id: 7, name: 'Candy Wrapper', category: 'trash', image: '🍬' },
  { id: 8, name: 'Pizza Box', category: 'compost', image: '🍕' }
]

const bins = [
  { id: 1, type: 'recycling', name: 'Recycling Bin', color: '#dc143c' }, // crimson
  { id: 2, type: 'compost', name: 'Compost Bin', color: '#8b4513' }, // saddlebrown
  { id: 3, type: 'hazardous', name: 'Hazardous Bin', color: '#ff8c00' }, // darkorange
  { id: 4, type: 'trash', name: 'Trash Bin', color: '#696969' } // dimgray
]

export default function App() {
  const [score, setScore] = useState(0)
  const [currentItem, setCurrentItem] = useState(null)
  const [gameOver, setGameOver] = useState(false)
  const [showConfetti, setShowConfetti] = useState(false)
  const [leaderboard, setLeaderboard] = useState([])
  const [playerName, setPlayerName] = useState('')
  const [nameSubmitted, setNameSubmitted] = useState(false)
  const [round, setRound] = useState(1)
  const MAX_ROUNDS = 5

  useEffect(() => {
    if (localStorage.getItem('leaderboard')) {
      setLeaderboard(JSON.parse(localStorage.getItem('leaderboard')))
    }
    selectRandomItem()
  }, [])

  const selectRandomItem = () => {
    const randomIndex = Math.floor(Math.random() * wasteItems.length)
    setCurrentItem(wasteItems[randomIndex])
  }

  const handleBinClick = (binType) => {
    if (!currentItem) return

    if (binType === currentItem.category) {
      // Correct bin
      const newScore = score + 10
      setScore(newScore)
      setShowConfetti(true)
      setTimeout(() => setShowConfetti(false), 2000)
      
      // Show feedback
      alert(`Great job! ${currentItem.name} goes in the ${binType} bin!`)
    } else {
      // Wrong bin
      alert(`Oops! ${currentItem.name} doesn't belong in the ${binType} bin. Try again!`)
    }

    // Next round or end game
    if (round < MAX_ROUNDS) {
      setRound(round + 1)
      selectRandomItem()
    } else {
      setGameOver(true)
    }
  }

  const handleNameSubmit = (e) => {
    e.preventDefault()
    if (!playerName.trim()) return
    
    const newEntry = { name: playerName, score }
    const updatedLeaderboard = [...leaderboard, newEntry]
      .sort((a, b) => b.score - a.score)
      .slice(0, 10)
    
    setLeaderboard(updatedLeaderboard)
    localStorage.setItem('leaderboard', JSON.stringify(updatedLeaderboard))
    setNameSubmitted(true)
  }

  const resetGame = () => {
    setScore(0)
    setRound(1)
    setGameOver(false)
    setNameSubmitted(false)
    setPlayerName('')
    selectRandomItem()
  }

  return (
    <div className="app">
      {showConfetti && <ReactConfetti />}
      
      <header>
        <h1>♻️ Recycle Rangers ♻️</h1>
        <p>Help sort the waste and save the planet!</p>
      </header>

      {!nameSubmitted && !gameOver && (
        <div className="game-area">
          <div className="score-board">
            <h2>Round: {round}/{MAX_ROUNDS}</h2>
            <h2>Score: {score}</h2>
          </div>
          
          <div className="current-item">
            {currentItem && (
              <>
                <div className="item-image" style={{ fontSize: '4rem' }}>
                  {currentItem.image}
                </div>
                <h3>{currentItem.name}</h3>
                <p>Where does this belong?</p>
              </>
            )}
          </div>
          
          <div className="bins-container">
            {bins.map(bin => (
              <button
                key={bin.id}
                className="bin"
                style={{ backgroundColor: bin.color }}
                onClick={() => handleBinClick(bin.type)}
              >
                {bin.name}
              </button>
            ))}
          </div>
        </div>
      )}

      {gameOver && !nameSubmitted && (
        <div className="game-over">
          <h2>Game Over!</h2>
          <p>Your final score: {score}</p>
          
          <form onSubmit={handleNameSubmit}>
            <label>
              Enter your name for the leaderboard:
              <input
                type="text"
                value={playerName}
                onChange={(e) => setPlayerName(e.target.value)}
                maxLength="20"
                required
              />
            </label>
            <button type="submit">Submit</button>
          </form>
        </div>
      )}

      {nameSubmitted && (
        <div className="leaderboard">
          <h2>🏆 Leaderboard 🏆</h2>
          <ol>
            {leaderboard.map((entry, index) => (
              <li key={index}>
                {entry.name}: {entry.score} points
              </li>
            ))}
          </ol>
          <button onClick={resetGame}>Play Again</button>
        </div>
      )}
    </div>
  )
}
