const express = require('express')
const cors = require('cors')
const sqlite3 = require('sqlite3').verbose()
const db = new sqlite3.Database('./trivia.db')

const app = express()
app.use(cors())
app.use(express.json())

// Create tables
db.serialize(() => {
  db.run(`
    CREATE TABLE IF NOT EXISTS contests (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      title TEXT NOT NULL,
      description TEXT,
      start_time DATETIME,
      end_time DATETIME
    )
  `)

  db.run(`
    CREATE TABLE IF NOT EXISTS questions (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      contest_id INTEGER,
      question_text TEXT NOT NULL,
      option_a TEXT,
      option_b TEXT,
      option_c TEXT,
      option_d TEXT,
      correct_answer TEXT,
      points INTEGER,
      FOREIGN KEY (contest_id) REFERENCES contests(id)
    )
  `)

  db.run(`
    CREATE TABLE IF NOT EXISTS users (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      username TEXT UNIQUE NOT NULL,
      email TEXT UNIQUE NOT NULL,
      password TEXT NOT NULL
    )
  `)

  db.run(`
    CREATE TABLE IF NOT EXISTS user_answers (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      user_id INTEGER,
      question_id INTEGER,
      answer TEXT,
      is_correct BOOLEAN,
      timestamp DATETIME DEFAULT CURRENT_TIMESTAMP,
      FOREIGN KEY (user_id) REFERENCES users(id),
      FOREIGN KEY (question_id) REFERENCES questions(id)
    )
  `)
})

// API Routes
app.get('/api/contests', (req, res) => {
  db.all('SELECT * FROM contests', [], (err, rows) => {
    if (err) {
      return res.status(500).json({ error: err.message })
    }
    res.json(rows)
  })
})

app.get('/api/contest/:id', (req, res) => {
  const contestId = req.params.id
  db.get('SELECT * FROM contests WHERE id = ?', [contestId], (err, contest) => {
    if (err) {
      return res.status(500).json({ error: err.message })
    }
    db.all(
      'SELECT * FROM questions WHERE contest_id = ?',
      [contestId],
      (err, questions) => {
        if (err) {
          return res.status(500).json({ error: err.message })
        }
        res.json({ ...contest, questions })
      }
    )
  })
})

app.post('/api/submit-answer', (req, res) => {
  const { userId, questionId, answer } = req.body
  db.get(
    'SELECT correct_answer FROM questions WHERE id = ?',
    [questionId],
    (err, question) => {
      if (err) {
        return res.status(500).json({ error: err.message })
      }
      const isCorrect = answer === question.correct_answer
      db.run(
        'INSERT INTO user_answers (user_id, question_id, answer, is_correct) VALUES (?, ?, ?, ?)',
        [userId, questionId, answer, isCorrect],
        function (err) {
          if (err) {
            return res.status(500).json({ error: err.message })
          }
          res.json({ isCorrect })
        }
      )
    }
  )
})

app.get('/api/leaderboard', (req, res) => {
  db.all(
    `SELECT u.username, SUM(q.points) as score
     FROM user_answers ua
     JOIN questions q ON ua.question_id = q.id
     JOIN users u ON ua.user_id = u.id
     WHERE ua.is_correct = 1
     GROUP BY u.id
     ORDER BY score DESC
     LIMIT 10`,
    [],
    (err, rows) => {
      if (err) {
        return res.status(500).json({ error: err.message })
      }
      res.json(rows)
    }
  )
})

const PORT = process.env.PORT || 5000
app.listen(PORT, () => {
  console.log(`Server running on port ${PORT}`)
})
