import { useState, useEffect } from 'react'
import { useParams, useNavigate } from 'react-router-dom'
import { FaEdit, FaTrash, FaPlus, FaShare } from 'react-icons/fa'
import '../App.css'

function Dashboard() {
  const { userId } = useParams()
  const navigate = useNavigate()
  const [links, setLinks] = useState([])
  const [title, setTitle] = useState('')
  const [url, setUrl] = useState('')
  const [editingId, setEditingId] = useState(null)

  useEffect(() => {
    const savedLinks = localStorage.getItem(`links-${userId}`)
    if (savedLinks) {
      setLinks(JSON.parse(savedLinks))
    }
  }, [userId])

  const saveLinks = (newLinks) => {
    setLinks(newLinks)
    localStorage.setItem(`links-${userId}`, JSON.stringify(newLinks))
  }

  const handleAddLink = () => {
    if (!title || !url) return

    const newLink = {
      id: Date.now(),
      title,
      url: url.startsWith('http') ? url : `https://${url}`,
    }

    saveLinks([...links, newLink])
    setTitle('')
    setUrl('')
  }

  const handleEditLink = (id) => {
    const linkToEdit = links.find((link) => link.id === id)
    if (linkToEdit) {
      setTitle(linkToEdit.title)
      setUrl(linkToEdit.url)
      setEditingId(id)
    }
  }

  const handleUpdateLink = () => {
    if (!title || !url || !editingId) return

    const updatedLinks = links.map((link) =>
      link.id === editingId
        ? {
            ...link,
            title,
            url: url.startsWith('http') ? url : `https://${url}`,
          }
        : link
    )

    saveLinks(updatedLinks)
    setTitle('')
    setUrl('')
    setEditingId(null)
  }

  const handleDeleteLink = (id) => {
    const updatedLinks = links.filter((link) => link.id !== id)
    saveLinks(updatedLinks)
  }

  const handleShare = () => {
    const shareUrl = `${window.location.origin}/dashboard/${userId}`
    navigator.clipboard.writeText(shareUrl)
    alert('Link copied to clipboard!')
  }

  return (
    <div className="app-container">
      <h1>Your Link Tree</h1>
      <p>User ID: {userId}</p>

      <div>
        <input
          type="text"
          className="input-field"
          placeholder="Link Title"
          value={title}
          onChange={(e) => setTitle(e.target.value)}
        />
        <input
          type="text"
          className="input-field"
          placeholder="URL (https://example.com)"
          value={url}
          onChange={(e) => setUrl(e.target.value)}
        />
        {editingId ? (
          <button className="button" onClick={handleUpdateLink}>
            <FaEdit /> Update Link
          </button>
        ) : (
          <button className="button" onClick={handleAddLink}>
            <FaPlus /> Add Link
          </button>
        )}
        <button className="button" onClick={handleShare}>
          <FaShare /> Share Your Link Tree
        </button>
      </div>

      <div>
        {links.map((link) => (
          <div key={link.id} style={{ display: 'flex', alignItems: 'center' }}>
            <a
              href={link.url}
              target="_blank"
              rel="noopener noreferrer"
              className="link-card"
            >
              {link.title}
            </a>
            <button
              className="button"
              onClick={() => handleEditLink(link.id)}
              style={{ marginLeft: '10px' }}
            >
              <FaEdit />
            </button>
            <button
              className="button"
              onClick={() => handleDeleteLink(link.id)}
            >
              <FaTrash />
            </button>
          </div>
        ))}
      </div>
    </div>
  )
}

export default Dashboard
