import express from 'express'
import sqlite3 from 'sqlite3'
import { open } from 'sqlite'
import cors from 'cors'
import { v4 as uuidv4 } from 'uuid'
import path from 'path'
import fs from 'fs'
import multer from 'multer'

const app = express()
app.use(cors())
app.use(express.json())

// Configure file uploads
const upload = multer({ dest: 'uploads/' })
if (!fs.existsSync('uploads')) {
  fs.mkdirSync('uploads')
}

let db
async function initializeDatabase() {
  db = await open({
    filename: './construction.db',
    driver: sqlite3.Database
  })

  await db.exec(`
    CREATE TABLE IF NOT EXISTS projects (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      name TEXT NOT NULL,
      description TEXT,
      start_date TEXT,
      end_date TEXT,
      status TEXT DEFAULT 'planned',
      budget REAL,
      created_at TEXT DEFAULT CURRENT_TIMESTAMP
    );
    
    CREATE TABLE IF NOT EXISTS documents (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      project_id INTEGER,
      name TEXT NOT NULL,
      type TEXT,
      file_path TEXT,
      upload_date TEXT DEFAULT CURRENT_TIMESTAMP,
      FOREIGN KEY(project_id) REFERENCES projects(id)
    );
    
    CREATE TABLE IF NOT EXISTS tasks (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      project_id INTEGER,
      name TEXT NOT NULL,
      description TEXT,
      start_date TEXT,
      end_date TEXT,
      status TEXT DEFAULT 'not_started',
      priority TEXT DEFAULT 'medium',
      FOREIGN KEY(project_id) REFERENCES projects(id)
    );
    
    CREATE TABLE IF NOT EXISTS resources (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      name TEXT NOT NULL,
      type TEXT CHECK(type IN ('material', 'equipment', 'labor')),
      quantity INTEGER,
      cost REAL,
      availability TEXT,
      notes TEXT
    );
    
    CREATE TABLE IF NOT EXISTS time_entries (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      project_id INTEGER,
      task_id INTEGER,
      user_id INTEGER,
      hours REAL,
      date TEXT DEFAULT CURRENT_TIMESTAMP,
      description TEXT,
      FOREIGN KEY(project_id) REFERENCES projects(id),
      FOREIGN KEY(task_id) REFERENCES tasks(id)
    );
    
    CREATE TABLE IF NOT EXISTS users (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      username TEXT UNIQUE NOT NULL,
      password TEXT NOT NULL,
      role TEXT DEFAULT 'user',
      name TEXT
    );
  `)
}

initializeDatabase()

// Helper function for error handling
const handleError = (res, error) => {
  console.error(error)
  res.status(500).json({ error: error.message })
}

// Projects API
app.get('/api/projects', async (req, res) => {
  try {
    const projects = await db.all('SELECT * FROM projects ORDER BY created_at DESC')
    res.json(projects)
  } catch (error) {
    handleError(res, error)
  }
})

app.post('/api/projects', async (req, res) => {
  try {
    const { name, description, start_date, end_date, budget } = req.body
    const result = await db.run(
      'INSERT INTO projects (name, description, start_date, end_date, budget) VALUES (?, ?, ?, ?, ?)',
      [name, description, start_date, end_date, budget]
    )
    const project = await db.get('SELECT * FROM projects WHERE id = ?', result.lastID)
    res.status(201).json(project)
  } catch (error) {
    handleError(res, error)
  }
})

app.get('/api/projects/:id', async (req, res) => {
  try {
    const project = await db.get('SELECT * FROM projects WHERE id = ?', req.params.id)
    if (project) {
      res.json(project)
    } else {
      res.status(404).json({ error: 'Project not found' })
    }
  } catch (error) {
    handleError(res, error)
  }
})

app.put('/api/projects/:id', async (req, res) => {
  try {
    const { name, description, start_date, end_date, budget, status } = req.body
    await db.run(
      'UPDATE projects SET name = ?, description = ?, start_date = ?, end_date = ?, budget = ?, status = ? WHERE id = ?',
      [name, description, start_date, end_date, budget, status, req.params.id]
    )
    res.status(204).end()
  } catch (error) {
    handleError(res, error)
  }
})

app.delete('/api/projects/:id', async (req, res) => {
  try {
    await db.run('DELETE FROM projects WHERE id = ?', req.params.id)
    res.status(204).end()
  } catch (error) {
    handleError(res, error)
  }
})

// Documents API
app.post('/api/projects/:id/documents', upload.single('file'), async (req, res) => {
  try {
    if (!req.file) {
      return res.status(400).json({ error: 'No file uploaded' })
    }

    const { originalname, mimetype, filename } = req.file
    const filePath = path.join('uploads', filename)
    const newFileName = `${uuidv4()}${path.extname(originalname)}`
    const newFilePath = path.join('uploads', newFileName)

    fs.renameSync(filePath, newFilePath)

    const result = await db.run(
      'INSERT INTO documents (project_id, name, type, file_path) VALUES (?, ?, ?, ?)',
      [req.params.id, originalname, mimetype, newFileName]
    )

    const document = await db.get('SELECT * FROM documents WHERE id = ?', result.lastID)
    res.status(201).json(document)
  } catch (error) {
    handleError(res, error)
  }
})

app.get('/api/projects/:id/documents', async (req, res) => {
  try {
    const documents = await db.all(
      'SELECT * FROM documents WHERE project_id = ? ORDER BY upload_date DESC',
      req.params.id
    )
    res.json(documents)
  } catch (error) {
    handleError(res, error)
  }
})

// Tasks API
app.get('/api/projects/:id/tasks', async (req, res) => {
  try {
    const tasks = await db.all(
      'SELECT * FROM tasks WHERE project_id = ? ORDER BY start_date ASC',
      req.params.id
    )
    res.json(tasks)
  } catch (error) {
    handleError(res, error)
  }
})

app.post('/api/projects/:id/tasks', async (req, res) => {
  try {
    const { name, description, start_date, end_date, priority } = req.body
    const result = await db.run(
      'INSERT INTO tasks (project_id, name, description, start_date, end_date, priority) VALUES (?, ?, ?, ?, ?, ?)',
      [req.params.id, name, description, start_date, end_date, priority]
    )
    const task = await db.get('SELECT * FROM tasks WHERE id = ?', result.lastID)
    res.status(201).json(task)
  } catch (error) {
    handleError(res, error)
  }
})

// Resources API
app.get('/api/resources', async (req, res) => {
  try {
    const resources = await db.all('SELECT * FROM resources ORDER BY name ASC')
    res.json(resources)
  } catch (error) {
    handleError(res, error)
  }
})

app.post('/api/resources', async (req, res) => {
  try {
    const { name, type, quantity, cost, availability, notes } = req.body
    const result = await db.run(
      'INSERT INTO resources (name, type, quantity, cost, availability, notes) VALUES (?, ?, ?, ?, ?, ?)',
      [name, type, quantity, cost, availability, notes]
    )
    const resource = await db.get('SELECT * FROM resources WHERE id = ?', result.lastID)
    res.status(201).json(resource)
  } catch (error) {
    handleError(res, error)
  }
})

// Time Entries API
app.post('/api/time-entries', async (req, res) => {
  try {
    const { project_id, task_id, user_id, hours, description } = req.body
    const result = await db.run(
      'INSERT INTO time_entries (project_id, task_id, user_id, hours, description) VALUES (?, ?, ?, ?, ?)',
      [project_id, task_id, user_id, hours, description]
    )
    const entry = await db.get('SELECT * FROM time_entries WHERE id = ?', result.lastID)
    res.status(201).json(entry)
  } catch (error) {
    handleError(res, error)
  }
})

// Serve uploaded files
app.get('/api/documents/:filename', (req, res) => {
  const filePath = path.join(__dirname, 'uploads', req.params.filename)
  if (fs.existsSync(filePath)) {
    res.sendFile(filePath)
  } else {
    res.status(404).json({ error: 'File not found' })
  }
})

const PORT = process.env.PORT || 5000
app.listen(PORT, () => {
  console.log(`Server running on port ${PORT}`)
})
