require('dotenv').config()
const express = require('express')
const cors = require('cors')
const bodyParser = require('body-parser')
const sqlite3 = require('sqlite3').verbose()
const bcrypt = require('bcrypt')
const jwt = require('jsonwebtoken')

const app = express()
const PORT = process.env.PORT || 5000

app.use(cors())
app.use(bodyParser.json())

// Database setup
const db = new sqlite3.Database('./database.db')

db.serialize(() => {
  db.run(`
    CREATE TABLE IF NOT EXISTS users (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      name TEXT NOT NULL,
      email TEXT UNIQUE NOT NULL,
      password TEXT NOT NULL,
      phone TEXT
    )
  `)

  db.run(`
    CREATE TABLE IF NOT EXISTS bookings (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      user_id INTEGER NOT NULL,
      transport_type TEXT NOT NULL,
      departure TEXT NOT NULL,
      destination TEXT NOT NULL,
      departure_time TEXT NOT NULL,
      arrival_time TEXT NOT NULL,
      seat_number TEXT NOT NULL,
      price REAL NOT NULL,
      status TEXT DEFAULT 'confirmed',
      FOREIGN KEY(user_id) REFERENCES users(id)
    )
  `)

  db.run(`
    CREATE TABLE IF NOT EXISTS transports (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      type TEXT NOT NULL,
      departure TEXT NOT NULL,
      destination TEXT NOT NULL,
      departure_time TEXT NOT NULL,
      arrival_time TEXT NOT NULL,
      available_seats INTEGER NOT NULL,
      price REAL NOT NULL
    )
  `)
})

// Auth middleware
const authenticateToken = (req, res, next) => {
  const token = req.headers['authorization']
  if (!token) return res.sendStatus(401)

  jwt.verify(token, process.env.JWT_SECRET, (err, user) => {
    if (err) return res.sendStatus(403)
    req.user = user
    next()
  })
}

// Routes
app.post('/api/register', async (req, res) => {
  const { name, email, password, phone } = req.body
  const hashedPassword = await bcrypt.hash(password, 10)

  db.run(
    'INSERT INTO users (name, email, password, phone) VALUES (?, ?, ?, ?)',
    [name, email, hashedPassword, phone],
    function (err) {
      if (err) {
        return res.status(400).json({ error: 'Email already exists' })
      }
      res.status(201).json({ id: this.lastID })
    }
  )
})

app.post('/api/login', (req, res) => {
  const { email, password } = req.body

  db.get('SELECT * FROM users WHERE email = ?', [email], async (err, user) => {
    if (err || !user) {
      return res.status(401).json({ error: 'Invalid credentials' })
    }

    const validPassword = await bcrypt.compare(password, user.password)
    if (!validPassword) {
      return res.status(401).json({ error: 'Invalid credentials' })
    }

    const token = jwt.sign(
      { id: user.id, email: user.email },
      process.env.JWT_SECRET,
      { expiresIn: '24h' }
    )

    res.json({ token, user: { id: user.id, name: user.name, email: user.email } })
  })
})

app.get('/api/transports', (req, res) => {
  const { departure, destination, type } = req.query
  let query = 'SELECT * FROM transports WHERE 1=1'
  const params = []

  if (departure) {
    query += ' AND departure = ?'
    params.push(departure)
  }
  if (destination) {
    query += ' AND destination = ?'
    params.push(destination)
  }
  if (type) {
    query += ' AND type = ?'
    params.push(type)
  }

  db.all(query, params, (err, transports) => {
    if (err) {
      return res.status(500).json({ error: err.message })
    }
    res.json(transports)
  })
})

app.post('/api/bookings', authenticateToken, (req, res) => {
  const { transport_id, seat_number } = req.body
  const user_id = req.user.id

  db.get('SELECT * FROM transports WHERE id = ?', [transport_id], (err, transport) => {
    if (err || !transport) {
      return res.status(400).json({ error: 'Transport not found' })
    }

    db.run(
      'INSERT INTO bookings (user_id, transport_type, departure, destination, departure_time, arrival_time, seat_number, price) VALUES (?, ?, ?, ?, ?, ?, ?, ?)',
      [
        user_id,
        transport.type,
        transport.departure,
        transport.destination,
        transport.departure_time,
        transport.arrival_time,
        seat_number,
        transport.price
      ],
      function (err) {
        if (err) {
          return res.status(500).json({ error: err.message })
        }

        // Update available seats
        db.run(
          'UPDATE transports SET available_seats = available_seats - 1 WHERE id = ?',
          [transport_id]
        )

        res.status(201).json({ bookingId: this.lastID })
      }
    )
  })
})

app.get('/api/bookings', authenticateToken, (req, res) => {
  db.all(
    'SELECT * FROM bookings WHERE user_id = ?',
    [req.user.id],
    (err, bookings) => {
      if (err) {
        return res.status(500).json({ error: err.message })
      }
      res.json(bookings)
    }
  )
})

app.listen(PORT, () => {
  console.log(`Server running on port ${PORT}`)
})
