const express = require('express')
const cors = require('cors')
const bodyParser = require('body-parser')
const { v4: uuidv4 } = require('uuid')
const sqlite3 = require('sqlite3').verbose()
const path = require('path')

const app = express()
app.use(cors())
app.use(bodyParser.json())

// Initialize database
const db = new sqlite3.Database('./database.sqlite')

// Create tables if they don't exist
db.serialize(() => {
  db.run(`
    CREATE TABLE IF NOT EXISTS bookings (
      id TEXT PRIMARY KEY,
      name TEXT,
      email TEXT,
      phone TEXT,
      packageId TEXT,
      dates TEXT,
      players INTEGER,
      basePrice REAL,
      taxes REAL,
      total REAL,
      status TEXT,
      createdAt TEXT,
      confirmedAt TEXT
    )
  `)

  db.run(`
    CREATE TABLE IF NOT EXISTS customers (
      id TEXT PRIMARY KEY,
      name TEXT,
      email TEXT UNIQUE,
      phone TEXT,
      createdAt TEXT
    )
  `)

  db.run(`
    CREATE TABLE IF NOT EXISTS suppliers (
      id TEXT PRIMARY KEY,
      name TEXT,
      type TEXT,
      contactEmail TEXT,
      contactPhone TEXT,
      createdAt TEXT
    )
  `)
})

// Helper function for async/await database operations
function dbRun(sql, params) {
  return new Promise((resolve, reject) => {
    db.run(sql, params, function(err) {
      if (err) reject(err)
      else resolve(this)
    })
  })
}

function dbGet(sql, params) {
  return new Promise((resolve, reject) => {
    db.get(sql, params, (err, row) => {
      if (err) reject(err)
      else resolve(row)
    })
  })
}

function dbAll(sql, params) {
  return new Promise((resolve, reject) => {
    db.all(sql, params, (err, rows) => {
      if (err) reject(err)
      else resolve(rows)
    })
  })
}

// Customer endpoints
app.post('/api/booking', async (req, res) => {
  try {
    const bookingId = uuidv4()
    const customerId = uuidv4()
    const now = new Date().toISOString()
    
    // Calculate pricing (in a real app, this would be more complex)
    const basePrice = 1000 + (req.body.players * 200)
    const taxes = basePrice * 0.15
    const total = basePrice + taxes

    // Insert customer if not exists
    await dbRun(
      `INSERT OR IGNORE INTO customers (id, name, email, phone, createdAt) 
       VALUES (?, ?, ?, ?, ?)`,
      [customerId, req.body.name, req.body.email, req.body.phone, now]
    )

    // Create booking
    await dbRun(
      `INSERT INTO bookings (
        id, name, email, phone, packageId, dates, players, 
        basePrice, taxes, total, status, createdAt
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        bookingId,
        req.body.name,
        req.body.email,
        req.body.phone,
        req.body.packageId,
        req.body.dates,
        req.body.players,
        basePrice,
        taxes,
        total,
        'pending',
        now
      ]
    )

    const booking = await dbGet(
      `SELECT * FROM bookings WHERE id = ?`,
      [bookingId]
    )

    res.status(201).json(booking)
  } catch (err) {
    console.error('Booking error:', err)
    res.status(500).json({ error: 'Failed to create booking' })
  }
})

app.get('/api/booking/:id', async (req, res) => {
  try {
    const booking = await dbGet(
      `SELECT * FROM bookings WHERE id = ?`,
      [req.params.id]
    )
    
    if (booking) {
      res.json(booking)
    } else {
      res.status(404).json({ error: 'Booking not found' })
    }
  } catch (err) {
    console.error('Error fetching booking:', err)
    res.status(500).json({ error: 'Failed to fetch booking' })
  }
})

app.post('/api/booking/:id/confirm', async (req, res) => {
  try {
    const now = new Date().toISOString()
    await dbRun(
      `UPDATE bookings SET status = ?, confirmedAt = ? WHERE id = ?`,
      ['confirmed', now, req.params.id]
    )

    const booking = await dbGet(
      `SELECT * FROM bookings WHERE id = ?`,
      [req.params.id]
    )

    if (booking) {
      res.json(booking)
    } else {
      res.status(404).json({ error: 'Booking not found' })
    }
  } catch (err) {
    console.error('Confirmation error:', err)
    res.status(500).json({ error: 'Failed to confirm booking' })
  }
})

// Admin endpoints
app.get('/api/admin/bookings', async (req, res) => {
  try {
    const bookings = await dbAll(
      `SELECT * FROM bookings ORDER BY createdAt DESC`
    )
    res.json(bookings)
  } catch (err) {
    console.error('Error fetching bookings:', err)
    res.status(500).json({ error: 'Failed to fetch bookings' })
  }
})

app.get('/api/admin/customers', async (req, res) => {
  try {
    const customers = await dbAll(
      `SELECT * FROM customers ORDER BY createdAt DESC`
    )
    res.json(customers)
  } catch (err) {
    console.error('Error fetching customers:', err)
    res.status(500).json({ error: 'Failed to fetch customers' })
  }
})

app.get('/api/admin/suppliers', async (req, res) => {
  try {
    const suppliers = await dbAll(
      `SELECT * FROM suppliers ORDER BY createdAt DESC`
    )
    res.json(suppliers)
  } catch (err) {
    console.error('Error fetching suppliers:', err)
    res.status(500).json({ error: 'Failed to fetch suppliers' })
  }
})

// Serve static files in production
if (process.env.NODE_ENV === 'production') {
  app.use(express.static(path.join(__dirname, '../dist')))
  app.get('*', (req, res) => {
    res.sendFile(path.join(__dirname, '../dist', 'index.html'))
  })
}

const PORT = process.env.PORT || 5000
app.listen(PORT, () => {
  console.log(`Server running on port ${PORT}`)
})

process.on('SIGINT', () => {
  db.close()
  process.exit()
})
