import './style.css';

interface Player {
  id: number;
  name: string;
  position: string;
  battingAverage: number;
  homeRuns: number;
  rbi: number;
}

interface Team {
  id: number;
  name: string;
  players: Player[];
  budget: number;
}

class BaseballSimulator {
  private teams: Team[] = [];
  private currentView: string = 'dashboard';

  constructor() {
    this.init();
  }

  private init(): void {
    this.render();
    // Initialize with sample data
    this.teams = [
      {
        id: 1,
        name: 'Home Team',
        players: this.generatePlayers(1),
        budget: 5000000
      },
      {
        id: 2,
        name: 'Away Team',
        players: this.generatePlayers(2),
        budget: 4500000
      }
    ];
  }

  private generatePlayers(teamId: number): Player[] {
    const positions = ['Pitcher', 'Catcher', '1B', '2B', '3B', 'SS', 'LF', 'CF', 'RF'];
    return positions.map((position, index) => ({
      id: teamId * 100 + index,
      name: `Player ${teamId}-${index + 1}`,
      position,
      battingAverage: Math.random() * 0.3 + 0.1,
      homeRuns: Math.floor(Math.random() * 30),
      rbi: Math.floor(Math.random() * 100)
    }));
  }

  private render(): void {
    const app = document.getElementById('app');
    if (!app) return;

    app.innerHTML = `
      <h1>Baseball Simulator</h1>
      <div class="controls">
        <button id="dashboardBtn">Dashboard</button>
        <button id="teamsBtn">Teams</button>
        <button id="simulateBtn">Simulate Game</button>
      </div>
      <div id="content"></div>
    `;

    // Add event listeners
    document.getElementById('dashboardBtn')?.addEventListener('click', () => this.showView('dashboard'));
    document.getElementById('teamsBtn')?.addEventListener('click', () => this.showView('teams'));
    document.getElementById('simulateBtn')?.addEventListener('click', () => this.simulateGame());
  }

  private showView(view: string): void {
    this.currentView = view;
    const content = document.getElementById('content');
    if (!content) return;

    if (view === 'dashboard') {
      content.innerHTML = '<h2>Dashboard</h2><p>Welcome to Baseball Simulator</p>';
    } else if (view === 'teams') {
      content.innerHTML = this.renderTeamsView();
    }
  }

  private renderTeamsView(): string {
    return `
      <h2>Teams</h2>
      ${this.teams.map(team => `
        <div class="team-card">
          <h3>${team.name}</h3>
          <p>Budget: $${team.budget.toLocaleString()}</p>
          <h4>Players:</h4>
          <ul>
            ${team.players.map(player => `
              <li>
                ${player.name} (${player.position}) - 
                BA: ${player.battingAverage.toFixed(3)}, 
                HR: ${player.homeRuns}, 
                RBI: ${player.rbi}
              </li>
            `).join('')}
          </ul>
        </div>
      `).join('')}
    `;
  }

  private simulateGame(): void {
    const content = document.getElementById('content');
    if (!content) return;

    content.innerHTML = '<h2>Game Simulation</h2><p>Simulating game...</p>';
    // In a real implementation, this would be more complex
    setTimeout(() => {
      const homeScore = Math.floor(Math.random() * 10);
      const awayScore = Math.floor(Math.random() * 10);
      content.innerHTML = `
        <h2>Game Results</h2>
        <p>${this.teams[0].name}: ${homeScore}</p>
        <p>${this.teams[1].name}: ${awayScore}</p>
        <p>Winner: ${homeScore > awayScore ? this.teams[0].name : this.teams[1].name}</p>
      `;
    }, 2000);
  }
}

new BaseballSimulator();