import express = require('express');
import cors from 'cors';

const app = express();
const PORT = 3000;

app.use(cors());
app.use(express.json());

// Type definitions
interface Post {
  id: number;
  title: string;
  content: string;
  category: string;
  replies: Reply[];
}

interface Reply {
  id: number;
  content: string;
  author: string;
  createdAt: string;
}

// Temporary in-memory data store
let posts: Post[] = [
  { 
    id: 1, 
    title: 'First Post', 
    content: 'This is the first post', 
    category: 'General',
    replies: [
      { id: 1, content: 'First reply', author: 'user1', createdAt: new Date().toISOString() }
    ]
  },
  { 
    id: 2, 
    title: 'Second Post', 
    content: 'This is the second post', 
    category: 'Questions',
    replies: []
  }
];

interface User {
  id: number;
  username: string;
  email: string;
  joinDate: string;
  posts: number[];
  replies: number[];
}

let users: User[] = [
  {
    id: 1,
    username: 'testuser',
    email: 'test@example.com',
    joinDate: new Date().toISOString(),
    posts: [1],
    replies: [1]
  }
];

// API routes
app.get('/api/posts', (req: express.Request, res: express.Response) => {
  res.json(posts);
});

app.get('/api/posts/:id', (req: express.Request, res: express.Response) => {
  const postId = parseInt(req.params.id);
  const post = posts.find(p => p.id === postId);
  if (!post) {
    res.status(404).send('Post not found');
    return;
  }
  res.json(post);
});

app.post('/api/posts', (req: express.Request, res: express.Response) => {
  const newPost: Post = {
    id: posts.length + 1,
    title: req.body.title,
    content: req.body.content,
    category: req.body.category,
    replies: []
  };
  posts.push(newPost);
  res.status(201).json(newPost);
});

app.post('/api/posts/:id/replies', (req: express.Request, res: express.Response) => {
  const postId = parseInt(req.params.id);
  const post = posts.find(p => p.id === postId);
  if (!post) {
    res.status(404).send('Post not found');
    return;
  }
  
  const newReply: Reply = {
    id: post.replies.length + 1,
    content: req.body.content,
    author: req.body.author || 'Anonymous',
    createdAt: new Date().toISOString()
  };
  
  post.replies.push(newReply);
  res.status(201).json(newReply);
});

app.get('/api/user/current', (req: express.Request, res: express.Response) => {
  const user = users[0];
  const userData = {
    ...user,
    posts: posts.filter(p => user.posts.includes(p.id)),
    replies: posts.flatMap(p => 
      p.replies
        .filter(r => r.author === user.username)
        .map(r => ({
          id: r.id,
          postId: p.id,
          postTitle: p.title,
          content: r.content,
          createdAt: r.createdAt
        }))
    )
  };
  res.json(userData);
});

app.listen(PORT, () => {
  console.log(`Server running on http://localhost:${PORT}`);
});