import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { useParams, Link } from 'react-router-dom';

interface Post {
  id: number;
  title: string;
  content: string;
  category: string;
  replies?: Reply[];
}

interface Reply {
  id: number;
  content: string;
  author: string;
  createdAt: string;
}

const PostDetail: React.FC = () => {
  const { id } = useParams<{ id: string }>();
  const [post, setPost] = useState<Post | null>(null);
  const [replyContent, setReplyContent] = useState('');
  const [isLoading, setIsLoading] = useState(true);

  useEffect(() => {
    const fetchPost = async () => {
      try {
        const response = await axios.get(`/api/posts/${id}`);
        setPost(response.data);
        setIsLoading(false);
      } catch (error) {
        console.error('Error fetching post:', error);
        setIsLoading(false);
      }
    };

    fetchPost();
  }, [id]);

  const handleReplySubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    try {
      const response = await axios.post(`/api/posts/${id}/replies`, {
        content: replyContent,
        author: 'CurrentUser' // This would be replaced with actual user auth
      });
      setPost(prev => ({
        ...prev!,
        replies: [...(prev?.replies || []), response.data]
      }));
      setReplyContent('');
    } catch (error) {
      console.error('Error submitting reply:', error);
    }
  };

  if (isLoading) return <div>Loading...</div>;
  if (!post) return <div>Post not found</div>;

  return (
    <div className="post-detail">
      <div className="post-card">
        <h2>{post.title}</h2>
        <p>{post.content}</p>
        <span className="category">{post.category}</span>
        <Link to="/">
          <button>Back to Forum</button>
        </Link>
      </div>

      <div className="replies">
        <h3>Replies ({post.replies?.length || 0})</h3>
        {post.replies?.map(reply => (
          <div key={reply.id} className="reply-box">
            <p>{reply.content}</p>
            <small>By {reply.author} on {new Date(reply.createdAt).toLocaleString()}</small>
          </div>
        ))}

        <form onSubmit={handleReplySubmit}>
          <textarea
            value={replyContent}
            onChange={(e) => setReplyContent(e.target.value)}
            placeholder="Write your reply..."
            required
          />
          <button type="submit">Submit Reply</button>
        </form>
      </div>
    </div>
  );
};

export default PostDetail;