// Supply Chain Traceability App
type Product = {
    id: string;
    name: string;
    origin: string;
    timestamp: Date;
    history: Transaction[];
};

type Transaction = {
    from: string;
    to: string;
    timestamp: Date;
    location: string;
};

class SupplyChainApp {
    private products: Product[] = [];

    constructor() {
        this.init();
    }

    private init() {
        console.log('Supply Chain Traceability App initialized');
        this.renderHomePage();
    }

    private renderHomePage() {
        const app = document.getElementById('app');
        if (app) {
            app.innerHTML = `
                <h2>Welcome to Supply Chain Traceability</h2>
                <div class="actions">
                    <button id="registerBtn">Register New Product</button>
                    <button id="trackBtn">Track Product</button>
                </div>
            `;
            
            document.getElementById('registerBtn')?.addEventListener('click', () => this.showRegistrationForm());
            document.getElementById('trackBtn')?.addEventListener('click', () => this.showTrackingForm());
        }
    }

    private showRegistrationForm() {
        const app = document.getElementById('app');
        if (app) {
            app.innerHTML = `
                <h2>Product Registration</h2>
                <form id="productForm">
                    <div class="form-group">
                        <label for="productName">Product Name:</label>
                        <input type="text" id="productName" required>
                    </div>
                    <div class="form-group">
                        <label for="origin">Origin:</label>
                        <input type="text" id="origin" required>
                    </div>
                    <button type="submit">Register</button>
                </form>
            `;

            document.getElementById('productForm')?.addEventListener('submit', (e) => {
                e.preventDefault();
                this.registerProduct();
            });
        }
    }

    private registerProduct() {
        const nameInput = document.getElementById('productName') as HTMLInputElement;
        const originInput = document.getElementById('origin') as HTMLInputElement;

        const newProduct: Product = {
            id: `PROD-${Date.now()}`,
            name: nameInput.value,
            origin: originInput.value,
            timestamp: new Date(),
            history: []
        };

        this.products.push(newProduct);
        alert(`Product registered successfully! ID: ${newProduct.id}`);
        this.renderHomePage();
    }

    private showTrackingForm() {
        const app = document.getElementById('app');
        if (app) {
            app.innerHTML = `
                <h2>Product Tracking</h2>
                <form id="trackingForm">
                    <div class="form-group">
                        <label for="productId">Product ID:</label>
                        <input type="text" id="productId" required>
                    </div>
                    <button type="submit">Track</button>
                </form>
                <div id="trackingResult"></div>
            `;

            document.getElementById('trackingForm')?.addEventListener('submit', (e) => {
                e.preventDefault();
                this.trackProduct();
            });
        }
    }

    private trackProduct() {
        const productIdInput = document.getElementById('productId') as HTMLInputElement;
        const resultDiv = document.getElementById('trackingResult') as HTMLDivElement;
        
        const product = this.products.find(p => p.id === productIdInput.value);
        
        if (product) {
            resultDiv.innerHTML = `
                <h3>Product Details</h3>
                <p><strong>ID:</strong> ${product.id}</p>
                <p><strong>Name:</strong> ${product.name}</p>
                <p><strong>Origin:</strong> ${product.origin}</p>
                <p><strong>Registration Date:</strong> ${product.timestamp.toLocaleString()}</p>
            `;
        } else {
            resultDiv.innerHTML = '<p class="error">Product not found. Please check the ID.</p>';
        }
    }
}

// Initialize the app when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    new SupplyChainApp();
});