import React from 'react';
import ReactDOM from 'react-dom';
import { Chart, BarElement, CategoryScale, LinearScale } from 'chart.js';
import { Bar } from 'react-chartjs-2';

Chart.register(BarElement, CategoryScale, LinearScale);

class RiskAssessment extends React.Component {
  constructor(props) {
    super(props);
    this.state = {
      selectedItems: [],
      riskData: [],
      chartData: {
        labels: [],
        datasets: [{
          label: 'Risk Level',
          data: [],
          backgroundColor: 'steelblue'
        }]
      }
    };
  }

  componentDidMount() {
    // Simulate loading risk assessment items from backend
    this.loadAssessmentItems();
  }

  loadAssessmentItems = async () => {
    try {
      // In a real app, this would be an API call
      const mockItems = [
        { id: 1, name: 'Financial Risk', defaultRisk: 3 },
        { id: 2, name: 'Operational Risk', defaultRisk: 2 },
        { id: 3, name: 'Compliance Risk', defaultRisk: 4 }
      ];
      this.setState({ riskItems: mockItems });
    } catch (error) {
      console.error('Error loading assessment items:', error);
    }
  };

  handleItemSelect = (item) => {
    const { selectedItems, riskData } = this.state;
    const newSelectedItems = [...selectedItems];
    const newRiskData = [...riskData];
    
    const itemIndex = newSelectedItems.findIndex(i => i.id === item.id);
    if (itemIndex === -1) {
      newSelectedItems.push(item);
      newRiskData.push({ name: item.name, value: item.defaultRisk });
    } else {
      newSelectedItems.splice(itemIndex, 1);
      newRiskData.splice(itemIndex, 1);
    }

    this.setState({
      selectedItems: newSelectedItems,
      riskData: newRiskData,
      chartData: {
        labels: newRiskData.map(item => item.name),
        datasets: [{
          label: 'Risk Level',
          data: newRiskData.map(item => item.value),
          backgroundColor: 'steelblue'
        }]
      }
    });
  };

  generateReport = () => {
    // In a real app, this would generate and download a PDF report
    alert('Report generated for selected items!');
  };

  render() {
    const { riskItems = [], selectedItems, chartData } = this.state;

    return (
      <div>
        <h3>Select Risk Assessment Items</h3>
        <div className="item-list">
          {riskItems.map(item => (
            <div key={item.id}>
              <label>
                <input
                  type="checkbox"
                  checked={selectedItems.some(i => i.id === item.id)}
                  onChange={() => this.handleItemSelect(item)}
                />
                {item.name}
              </label>
            </div>
          ))}
        </div>

        {selectedItems.length > 0 && (
          <div className="chart-container">
            <h3>Risk Assessment Chart</h3>
            <div style={{ width: '600px', height: '400px' }}>
              <Bar data={chartData} options={{ responsive: true }} />
            </div>
            <button onClick={this.generateReport}>Generate Report</button>
          </div>
        )}
      </div>
    );
  }
}

ReactDOM.render(
  <RiskAssessment />,
  document.getElementById('chart-container')
);