import React, { useState } from 'react';
import ReactDOM from 'react-dom/client';
import './styles/main.css';
import { destinations, campsites } from './data/destinations';

const ReviewComponent = ({ review }: { review: Review }) => (
  <div className="review">
    <div className="review-header">
      <span className="review-user">{review.userName}</span>
      <span className="review-rating">{'★'.repeat(review.rating)}{'☆'.repeat(5 - review.rating)}</span>
      <span className="review-date">{review.date}</span>
    </div>
    <p className="review-comment">{review.comment}</p>
  </div>
);

const ReviewForm = ({ onSubmit }: { onSubmit: (review: Omit<Review, 'id'>) => void }) => {
  const [review, setReview] = useState({
    userName: '',
    rating: 5,
    comment: '',
    date: new Date().toISOString().split('T')[0]
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    onSubmit(review);
    setReview({
      userName: '',
      rating: 5,
      comment: '',
      date: new Date().toISOString().split('T')[0]
    });
  };

  return (
    <div className="card review-form">
      <h4>Add Your Review</h4>
      <form onSubmit={handleSubmit}>
        <div>
          <label>Your Name:</label>
          <input 
            type="text" 
            value={review.userName}
            onChange={(e) => setReview({...review, userName: e.target.value})}
            required
          />
        </div>
        <div>
          <label>Rating:</label>
          <select
            value={review.rating}
            onChange={(e) => setReview({...review, rating: parseInt(e.target.value)})}
          >
            {[1, 2, 3, 4, 5].map(num => (
              <option key={num} value={num}>{num} star{num !== 1 ? 's' : ''}</option>
            ))}
          </select>
        </div>
        <div>
          <label>Comment:</label>
          <textarea
            value={review.comment}
            onChange={(e) => setReview({...review, comment: e.target.value})}
            required
          />
        </div>
        <button type="submit">Submit Review</button>
      </form>
    </div>
  );
};

const DestinationCard = ({ destination, onBook }: { destination: typeof destinations[0], onBook: () => void }) => {
  const [showReviews, setShowReviews] = useState(false);
  const [showReviewForm, setShowReviewForm] = useState(false);
  const [currentReviews, setCurrentReviews] = useState(destination.reviews);

  const handleReviewSubmit = (newReview: Omit<Review, 'id'>) => {
    const reviewWithId = {
      ...newReview,
      id: Math.max(0, ...currentReviews.map(r => r.id)) + 1
    };
    setCurrentReviews([...currentReviews, reviewWithId]);
    setShowReviewForm(false);
  };

  return (
    <div className="card">
      <h3>{destination.name}, {destination.country}</h3>
      <p>{destination.description}</p>
      <div className="meta">
        <span>Rating: {currentReviews.length > 0 ? 
          (currentReviews.reduce((sum, review) => sum + review.rating, 0) / currentReviews.length).toFixed(1) : '0'}/5</span>
        <span>Price: {destination.priceRange}</span>
        <button onClick={onBook}>Book Now</button>
        <button onClick={() => setShowReviews(!showReviews)}>
          {showReviews ? 'Hide Reviews' : `Show Reviews (${currentReviews.length})`}
        </button>
        <button onClick={() => setShowReviewForm(!showReviewForm)}>
          {showReviewForm ? 'Cancel Review' : 'Add Review'}
        </button>
      </div>
      {showReviewForm && (
        <ReviewForm onSubmit={handleReviewSubmit} />
      )}
      {showReviews && (
        <div className="reviews-container">
          {currentReviews.map(review => (
            <ReviewComponent key={review.id} review={review} />
          ))}
        </div>
      )}
    </div>
  );
};

const CampsiteCard = ({ campsite, onBook }: { campsite: typeof campsites[0], onBook: () => void }) => {
  const [showReviews, setShowReviews] = useState(false);
  const [showReviewForm, setShowReviewForm] = useState(false);
  const [currentReviews, setCurrentReviews] = useState(campsite.reviews);

  const handleReviewSubmit = (newReview: Omit<Review, 'id'>) => {
    const reviewWithId = {
      ...newReview,
      id: Math.max(0, ...currentReviews.map(r => r.id)) + 1
    };
    setCurrentReviews([...currentReviews, reviewWithId]);
    setShowReviewForm(false);
  };

  return (
    <div className="card">
      <h3>{campsite.name}, {campsite.country}</h3>
      <p>{campsite.description}</p>
      <div className="meta">
        <span>Rating: {currentReviews.length > 0 ? 
          (currentReviews.reduce((sum, review) => sum + review.rating, 0) / currentReviews.length).toFixed(1) : '0'}/5</span>
        <span>Price: {campsite.priceRange}</span>
        <button onClick={onBook}>Book Now</button>
        <button onClick={() => setShowReviews(!showReviews)}>
          {showReviews ? 'Hide Reviews' : `Show Reviews (${currentReviews.length})`}
        </button>
        <button onClick={() => setShowReviewForm(!showReviewForm)}>
          {showReviewForm ? 'Cancel Review' : 'Add Review'}
        </button>
      </div>
      <ul>
        {campsite.amenities.map(amenity => (
          <li key={amenity}>{amenity}</li>
        ))}
      </ul>
      {showReviewForm && (
        <ReviewForm onSubmit={handleReviewSubmit} />
      )}
      {showReviews && (
        <div className="reviews-container">
          {currentReviews.map(review => (
            <ReviewComponent key={review.id} review={review} />
          ))}
        </div>
      )}
    </div>
  );
};

const BookingForm = ({ onClose, item }: { onClose: () => void, item: { name: string, type: string } }) => {
  const [dates, setDates] = useState({ start: '', end: '' });
  const [guests, setGuests] = useState(1);

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    alert(`Booking confirmed for ${item.name} (${item.type}) from ${dates.start} to ${dates.end} for ${guests} guests`);
    onClose();
  };

  return (
    <div className="card booking-form">
      <h3>Book {item.name}</h3>
      <form onSubmit={handleSubmit}>
        <div>
          <label>Start Date:</label>
          <input 
            type="date" 
            value={dates.start}
            onChange={(e) => setDates({...dates, start: e.target.value})}
            required
          />
        </div>
        <div>
          <label>End Date:</label>
          <input 
            type="date" 
            value={dates.end}
            onChange={(e) => setDates({...dates, end: e.target.value})}
            required
          />
        </div>
        <div>
          <label>Number of Guests:</label>
          <input 
            type="number" 
            min="1"
            value={guests}
            onChange={(e) => setGuests(parseInt(e.target.value))}
            required
          />
        </div>
        <div className="form-actions">
          <button type="submit">Confirm Booking</button>
          <button type="button" onClick={onClose}>Cancel</button>
        </div>
      </form>
    </div>
  );
};

const App = () => {
  const [showBookingForm, setShowBookingForm] = useState(false);
  const [currentBookingItem, setCurrentBookingItem] = useState<{name: string, type: string} | null>(null);

  const handleBookClick = (item: { name: string, type: string }) => {
    setCurrentBookingItem(item);
    setShowBookingForm(true);
  };

  const closeBookingForm = () => {
    setShowBookingForm(false);
    setCurrentBookingItem(null);
  };

  return (
    <div>
      <nav>
        <ul>
          <li><a href="#">Home</a></li>
          <li><a href="#">Destinations</a></li>
          <li><a href="#">Campsites</a></li>
          <li><a href="#">Bookings</a></li>
          <li><a href="#">Reviews</a></li>
        </ul>
      </nav>
      
      <main>
        <section className="hero">
          <h1>Explore Europe's Best Camping & Travel Spots</h1>
          <p>Discover hidden gems and popular destinations across the continent</p>
        </section>

        <section className="search">
          <h2>Find your perfect adventure</h2>
          <div className="search-controls">
            <select>
              <option>Select Country</option>
              <option>France</option>
              <option>Italy</option>
              <option>Spain</option>
              <option>Germany</option>
            </select>
            <button>Search</button>
          </div>
        </section>

        {showBookingForm && currentBookingItem && (
          <BookingForm onClose={closeBookingForm} item={currentBookingItem} />
        )}

        <section className="featured">
          <h2>Popular Destinations</h2>
          <div className="cards-container">
            {destinations.map(destination => (
              <DestinationCard 
                key={destination.id} 
                destination={destination}
                onBook={() => handleBookClick({ name: destination.name, type: 'destination' })}
              />
            ))}
          </div>
        </section>

        <section className="featured">
          <h2>Recommended Campsites</h2>
          <div className="cards-container">
            {campsites.map(campsite => (
              <CampsiteCard 
                key={campsite.id} 
                campsite={campsite}
                onBook={() => handleBookClick({ name: campsite.name, type: 'campsite' })}
              />
            ))}
          </div>
        </section>
      </main>
    </div>
  );
};

ReactDOM.createRoot(document.getElementById('root')!).render(
  <React.StrictMode>
    <App />
  </React.StrictMode>
);