from typing import List, Tuple

import numpy as np

from stego_benchmark.config import EvaluationType
from stego_benchmark.experiment import ExperimentResult


def calculate_success_rate(results: List[ExperimentResult]) -> float:
    """Calculate success rate for a set of results"""
    correct = sum(1 for r in results if r.is_correct)
    return (correct / len(results)) * 100 if results else 0.0


def calculate_response_lengths(results: List[ExperimentResult]) -> Tuple[float, float]:
    """Calculate mean and std of response lengths"""
    lengths = [len(r.task_response) for r in results]
    return np.mean(lengths), np.std(lengths)


def calculate_evaluation_scores(results: List[ExperimentResult]) -> Tuple[float, float]:
    """Calculate mean and std of evaluation scores"""
    scores = []
    for r in results:
        if r.evaluation and r.evaluation.type == EvaluationType.BASELINE_COMPARISON:
            scores.append(r.evaluation.info.aggregated_score)
    return np.mean(scores), np.std(scores) if scores else (0.0, 0.0)


def calculate_confidence_scores(results: List[ExperimentResult]) -> Tuple[float, float]:
    """Calculate mean and std of confidence scores for answer verification"""
    scores = []
    for r in results:
        if r.evaluation and r.evaluation.type == EvaluationType.ANSWER_VERIFICATION:
            scores.append(r.evaluation.info.confidence_score)
    return np.mean(scores), np.std(scores) if scores else (0.0, 0.0)
