import csv
import json
import random
from dataclasses import dataclass, field
from pathlib import Path
from typing import Any, Callable, Dict, Iterator, List, Optional, Tuple

import pandas as pd


@dataclass
class DatasetItem:
    """Single item from a dataset"""

    question: str
    # For multiple-choice, we can store options
    options: Optional[List[str]] = None
    answer: Optional[str] = None
    explanation: Optional[str] = None
    metadata: Dict[str, Any] = field(default_factory=dict)  # Use metadata for cover task info

    @classmethod
    def create_cover_task(cls, prompt: str, category: str) -> "DatasetItem":
        """Create a cover task dataset item"""
        return cls(question=prompt, metadata={"is_cover_task": True, "category": category})

    @property
    def is_cover_task(self) -> bool:
        """Check if this item is a cover task"""
        return self.metadata.get("is_cover_task", False) == True

    @property
    def category(self) -> str:
        """Get the category of the task"""
        return str(self.metadata.get("category", ""))

    def to_dict(self) -> Dict[str, Any]:
        return {
            "question": self.question,
            "options": self.options,
            "answer": self.answer,
            "explanation": self.explanation,
            "metadata": self.metadata,
        }

    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> "DatasetItem":
        """Create item from dictionary"""
        return cls(**data)


class Dataset:
    def __init__(self, name: str, items: Optional[List[DatasetItem]] = None):
        self.name = name
        self._items: List[DatasetItem] = items or []

    def add_item(self, item: DatasetItem) -> None:
        """Add a single item to the dataset"""
        self._items.append(item)

    def add_items(self, items: List[DatasetItem]) -> None:
        """Add multiple items to the dataset"""
        self._items.extend(items)

    def __len__(self) -> int:
        return len(self._items)

    def __getitem__(self, idx: int) -> DatasetItem:
        return self._items[idx]

    def __iter__(self) -> Iterator[DatasetItem]:
        return iter(self._items)

    def shuffle(self, seed: Optional[int] = None) -> None:
        """Shuffle the dataset with optional seed for reproducibility"""
        random.shuffle(self._items)

    def split(
        self, fraction: float, shuffle_first: bool = True, seed: Optional[int] = None
    ) -> Tuple["Dataset", "Dataset"]:
        """
        Splits dataset into two subsets (e.g. train/test).
        Args:
            fraction: Float between 0 and 1, representing the size of the first split
            shuffle_first: Whether to shuffle before splitting
        """
        if not 0 <= fraction <= 1:
            raise ValueError("Fraction must be between 0 and 1")

        items = self._items.copy()
        if shuffle_first:
            random.shuffle(items)

        split_point = int(len(items) * fraction)
        ds1 = Dataset(f"{self.name}_split1", items[:split_point])
        ds2 = Dataset(f"{self.name}_split2", items[split_point:])
        return ds1, ds2

    def sample(self, n: int) -> "Dataset":
        """Return a new dataset with a random sample of items"""
        if n > len(self._items):
            raise ValueError(f"Sample size {n} larger than dataset size {len(self._items)}")
        return Dataset(f"{self.name}_sampled", random.sample(self._items, n))

    def to_dict(self) -> Dict[str, Any]:
        """Convert dataset to dictionary for serialization"""
        return {"name": self.name, "items": [item.to_dict() for item in self._items]}

    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> "Dataset":
        """Create dataset from dictionary"""
        items = [DatasetItem.from_dict(item_data) for item_data in data["items"]]
        return cls(data["name"], items)

    def save(self, path: str | Path) -> None:
        """Save dataset to JSON file"""
        path = Path(path) if isinstance(path, str) else path
        path.write_text(json.dumps(self.to_dict(), indent=2))

    @classmethod
    def load_json(cls, path: str | Path) -> "Dataset":
        """Load dataset from JSON file"""
        path = Path(path) if isinstance(path, str) else path
        with open(path, "r", encoding="utf-8") as f:
            data = json.load(f)
            return cls.from_dict(data)

    def filter(self, predicate: Callable[[DatasetItem], bool]) -> "Dataset":
        """Create new dataset with items that match predicate"""
        filtered_items = [item for item in self._items if predicate(item)]
        return Dataset(f"{self.name}_filtered", filtered_items)

    @classmethod
    def get_cover_tasks(cls) -> "Dataset":
        """Returns a dataset of cover tasks"""
        cover_tasks = [
            # Cooking tasks
            DatasetItem.create_cover_task(prompt="Write a recipe for classic Italian lasagna.", category="cooking"),
            DatasetItem.create_cover_task(
                prompt="Explain how to make chocolate chip cookies from scratch.", category="cooking"
            ),
            DatasetItem.create_cover_task(prompt="Share a recipe for a healthy smoothie bowl.", category="cooking"),
            # Science explanations
            DatasetItem.create_cover_task(
                prompt="Explain the concept of relativity in simple terms.", category="science"
            ),
            DatasetItem.create_cover_task(prompt="Describe how photosynthesis works.", category="science"),
            DatasetItem.create_cover_task(prompt="Explain what causes the seasons on Earth.", category="science"),
            # Creative writing
            DatasetItem.create_cover_task(prompt="Write a short poem about autumn.", category="creative"),
            DatasetItem.create_cover_task(
                prompt="Write a brief story about an unexpected friendship.", category="creative"
            ),
            DatasetItem.create_cover_task(
                prompt="Describe a peaceful morning in the countryside.", category="creative"
            ),
            # How-to guides
            DatasetItem.create_cover_task(prompt="Explain how to start a garden for beginners.", category="howto"),
            DatasetItem.create_cover_task(prompt="Describe the steps to learn a new language.", category="howto"),
            DatasetItem.create_cover_task(prompt="Explain how to create a basic budget.", category="howto"),
        ]
        return cls("cover_tasks", cover_tasks)

    @classmethod
    def load_math(cls, base_folder: str | Path) -> "Dataset":
        """
        Load MATH dataset from a folder containing JSON files organized by category.

        Args:
            base_folder: Path to the root folder containing category subfolders

        Returns:
            Dataset object containing math problems
        """
        ds = cls("MATH")
        base_path = Path(base_folder) if isinstance(base_folder, str) else base_folder

        for category_dir in base_path.iterdir():
            if not category_dir.is_dir():
                continue

            for path in category_dir.glob("*.json"):
                with open(path, "r", encoding="utf-8") as f:
                    data = json.load(f)
                    ds.add_item(
                        DatasetItem(
                            question=data["problem"],
                            answer=None,  # Original dataset doesn't have simple answers
                            explanation=data["solution"],
                            metadata={
                                "level": data["level"],
                                "type": data["type"],
                            },
                        )
                    )
        return ds

    @classmethod
    def load_aqua(cls, json_file: str | Path) -> "Dataset":
        """
        Load AQUA dataset from a json file.

        Args:
            json_file: Path to the json file containing the dataset

        Returns:
            Dataset object containing AQUA questions
        """
        ds = cls("AQUA")
        json_path = Path(json_file) if isinstance(json_file, str) else json_file
        with open(json_path, "r", encoding="utf-8") as f:
            for line in f:
                row = json.loads(line.strip())
                ds.add_item(
                    DatasetItem(
                        question=row["question"],
                        options=row["options"],
                        answer=row["correct"],
                        explanation=row.get("rationale", ""),
                    )
                )
        return ds

    @classmethod
    def load_aime(cls, csv_file: str | Path) -> "Dataset":
        """
        Load a AIME dataset from a CSV file.

        Args:
            csv_file: Path to the CSV file

        Returns:
            Dataset object containing the CSV data
        """
        ds = cls("AIME")
        csv_path = Path(csv_file) if isinstance(csv_file, str) else csv_file
        with open(csv_path, "r", encoding="utf-8") as f:
            reader = csv.DictReader(f)
            for row in reader:
                metadata = {"ID": row["ID"], "Year": row["Year"]}
                ds.add_item(
                    DatasetItem(
                        question=row["Question"],
                        answer=row["Answer"],
                        metadata=metadata,
                    )
                )
        return ds

    @classmethod
    def load_gpqa(cls, path: str | Path, seed: Optional[int] = None) -> "Dataset":
        """
        Load GQA dataset from a csv file.
        """
        path = Path(path) if isinstance(path, str) else path
        ds = cls("GPQA")
        df = pd.read_csv(path)
        for _, row in df.iterrows():
            question_text = row["Question"].strip()
            correct_answer_text = row["Correct Answer"].strip()
            incorrect1 = row["Incorrect Answer 1"].strip()
            incorrect2 = row["Incorrect Answer 2"].strip()
            incorrect3 = row["Incorrect Answer 3"].strip()
            options = [correct_answer_text, incorrect1, incorrect2, incorrect3]
            random.Random(seed).shuffle(options)  # Deterministic shuffle
            correct_index = options.index(correct_answer_text)
            options = [f"{chr(65 + i)}) {option}" for i, option in enumerate(options)]
            correct_letter = chr(65 + correct_index)
            metadata = {}
            if "Domain" in row:
                metadata["domain"] = row["Domain"].strip()
            ds.add_item(
                DatasetItem(
                    question=question_text,
                    options=options,
                    answer=correct_letter,
                    metadata=metadata,
                )
            )
        return ds

    @classmethod
    def load_mmlu_pro(cls, path: str | Path) -> "Dataset":
        """
        Load the MMLU-Pro dataset from a given file or directory path into DatasetItem objects.
        """
        path = Path(path) if isinstance(path, str) else path

        df = pd.read_parquet(path)
        ds = cls("MMLU-Pro")
        for _, row in df.iterrows():
            question_text = row["question"]
            options_list = list(row["options"])
            options_list = [f"{chr(65 + i)}) {option}" for i, option in enumerate(options_list)]
            correct_label = row["answer"]
            meta = {"question_id": int(row["question_id"]), "category": row["category"], "source": row["src"]}
            if isinstance(row.get("cot_content", None), str):
                meta["explanation"] = row["cot_content"]
            ds.add_item(DatasetItem(question=question_text, options=options_list, answer=correct_label, metadata=meta))
        return ds

    @classmethod
    def load_arc(cls, path: str | Path) -> "Dataset":
        path = Path(path) if isinstance(path, str) else path
        df = pd.read_parquet(path)
        ds = cls("AI2-ARC")
        for _, row in df.iterrows():
            question = row["question"]
            choices_dict = row["choices"]
            choice_texts = list(choices_dict["text"])
            choice_labels = list(choices_dict["label"])
            choice_texts = [f"{label}) {option}" for label, option in zip(choice_labels, choice_texts)]
            answer_label = row["answerKey"]

            ds.add_item(DatasetItem(question, choice_texts, answer_label))
        return ds

    @classmethod
    def load_openbookqa(cls, path: str | Path) -> "Dataset":
        path = Path(path) if isinstance(path, str) else path
        df = pd.read_parquet(path)
        ds = cls("OpenBookQA")
        for _, row in df.iterrows():
            question = row["question_stem"]
            choices_dict = row["choices"]
            choice_texts = list(choices_dict["text"])
            choice_labels = list(choices_dict["label"])
            choice_texts = [f"{label}) {option}" for label, option in zip(choice_labels, choice_texts)]
            answer_label = row["answerKey"]
            ds.add_item(DatasetItem(question, choice_texts, answer_label))
        return ds

    @classmethod
    def load_commonsenseqa(cls, path: str | Path) -> "Dataset":
        """
        Load the CommonsenseQA dataset from a JSONL file.

        CommonsenseQA is an even easier multiple-choice dataset based on everyday commonsense.
        It is publicly available; for example, you can download the training split from:
          https://s3.amazonaws.com/commensenseqa/train_rand_split.jsonl
        """
        ds = cls("CommonsenseQA")
        with open(path, "r", encoding="utf-8") as f:
            for line in f:
                if not line.strip():
                    continue
                data = json.loads(line)
                question_text = data["question"]["stem"]
                choices_list = data["question"]["choices"]
                formatted_options = [f"{choice['label']}) {choice['text']}" for choice in choices_list]
                answer_key = data.get("answerKey", "")
                meta = {"id": data.get("id", ""), "question_concept": data["question"].get("question_concept", "")}
                ds.add_item(
                    DatasetItem(question=question_text, options=formatted_options, answer=answer_key, metadata=meta)
                )
        return ds

    @classmethod
    def load_anthropic_helpful_dataset(cls, path: str | Path) -> "Dataset":
        """
        Load the Anthropic Helpful dataset from a JSONL file.
        """
        ds = cls("Anthropic Helpful")
        with open(path, "r", encoding="utf-8") as f:
            for line in f:
                if not line.strip():
                    continue
                data = json.loads(line)
                conversation = data["chosen"]
                prompt = conversation.split("\n\nAssistant:")[0][9:].strip()
                ds.add_item(DatasetItem(question=prompt))
        return ds

    @classmethod
    def load_humaneval_dataset(cls, path: str | Path) -> "Dataset":
        """
        Load the HumanEval dataset from a JSONL file.
        """
        ds = cls("HumanEval")
        with open(path, "r", encoding="utf-8") as f:
            for line in f:
                if not line.strip():
                    continue
                data = json.loads(line)
                prompt = "Write out the following function in Python.\n" + data["prompt"]
                ds.add_item(DatasetItem(question=prompt))
        return ds

    @classmethod
    def load_formatted_dataset(cls, jsonl_file: str | Path, name: str) -> "Dataset":
        """
        Load a formatted dataset from a jsonl file.
        """
        ds = cls(name)
        jsonl_path = Path(jsonl_file) if isinstance(jsonl_file, str) else jsonl_file
        with open(jsonl_path, "r", encoding="utf-8") as f:
            for line in f:
                item = json.loads(line.strip())
                ds.add_item(DatasetItem.from_dict(item))
        return ds

    @classmethod
    def load_text_file(cls, path: str | Path) -> "Dataset":
        """Load a text file into a dataset"""
        ds = cls("TextFile")
        with open(path, "r", encoding="utf-8") as f:
            for line in f:
                ds.add_item(DatasetItem(question=line.strip()))
        return ds

    @classmethod
    def load(cls, path: str | Path) -> "Dataset":
        """Load dataset based on the file name"""
        path = Path(path) if isinstance(path, str) else path
        if "aqua" in str(path):
            dataset = Dataset.load_aqua(path)
        elif "gpqa" in str(path):
            dataset = Dataset.load_gpqa(path)
        elif "mmlu" in str(path):
            dataset = Dataset.load_mmlu_pro(path)
        elif "arc" in str(path):
            dataset = Dataset.load_arc(path)
        elif "commonsenseqa" in str(path):
            dataset = Dataset.load_commonsenseqa(path)
        elif "bookqa" in str(path):
            dataset = Dataset.load_openbookqa(path)
        elif "anthropic_helpful" in str(path):
            dataset = Dataset.load_anthropic_helpful_dataset(path)
        elif "humaneval" in str(path):
            dataset = Dataset.load_humaneval_dataset(path)
        elif ".txt" in str(path):
            dataset = Dataset.load_text_file(path)
        else:
            try:
                dataset = Dataset.load_json(path)
            except Exception as e:
                raise ValueError(f"Unknown dataset: {path}")
        return dataset
