import base64


def rotn_encode(text: str, n: int):
    result = ""
    for char in text:
        if char.isalpha():
            # Determine the case base (97 for lowercase, 65 for uppercase)
            base = 97 if char.islower() else 65
            # Convert to 0-25 range, add n, mod 26, then convert back
            rotated = chr((ord(char) - base + n) % 26 + base)
            result += rotated
        else:
            # Keep non-alphabetic characters unchanged
            result += char
    return result


def rotn_with_bars_encode(text: str, n: int):
    processed_text = rotn_encode(text, n)
    return "|".join(processed_text)


def base64_encode(text: str):
    encoded = base64.b64encode(text.encode("utf-8"))
    return encoded.decode("utf-8")


def base64_encode_multiple(text: str, count: int):
    for _ in range(count):
        text = base64_encode(text)
    return text


def hex_encode(text: str):
    return text.encode("utf-8").hex()


def hex_encode_multiple(text: str, count: int):
    for _ in range(count):
        text = hex_encode(text)
    return text


def hex_rot1_encode(text: str):
    hex_text = hex_encode(text).upper()
    out_str = ""
    for char in hex_text:
        if char.isdigit():
            if char == "9":
                out_str += "A"
            else:
                out_str += str(int(char) + 1)
        else:
            if char == "F":
                out_str += "0"
            else:
                out_str += chr(ord(char) + 1)
    return out_str


def substitution_encode(text: str, substitution_dict: dict):
    return "".join(substitution_dict[char] if char in substitution_dict else char for char in text)


def substitution_encode_multiple(text: str, substitution_dict: dict, count: int):
    for _ in range(count):
        text = substitution_encode(text, substitution_dict)
    return text


def binary_7bit_encode(text: str):
    return "".join(format(ord(char), "07b") for char in text)


def binary_8bit_encode(text: str):
    return "".join(format(ord(char), "08b") for char in text)


def binary_8bit_swap_encode(text: str):
    binary_str = binary_8bit_encode(text)
    return "".join("1" if bit == "0" else "0" for bit in binary_str)


def encode_multiple(text: str, count: int, method: str, params: dict = None):
    encode_dict = {
        "base64": base64_encode,
        "hex": hex_encode,
        "rotn": rotn_encode,
        "substitution": substitution_encode,
        "binary_7bit": binary_7bit_encode,
        "binary_8bit": binary_8bit_encode,
    }
    current_text = text
    for i in range(count):
        if method in encode_dict:
            current_text = encode_dict[method](current_text, **params)
        elif method == "mixed":
            current_text = encode_dict[params["mixed_method_list"][i]](current_text, **params["mixed_params_list"][i])
        else:
            raise ValueError(f"Invalid encoding method: {method}")

    return current_text


# Converts text to a binary string of 0s and 1s, where each word is a 0 if the first letter is in the first half of the alphabet and a 1 if it is in the second half
def decode_alphabet_half_word_binary(text: str):
    return "".join("0" if word[0].lower() in "abcdefghijklm" else "1" for word in text.split(" ") if word != "")


# Converts text to a binary string of 0s and 1s, where each sentence is a 0 if the first letter is in the first half of the alphabet and a 1 if it is in the second half
def decode_alphabet_half_sentence_binary(text: str):
    return "".join(
        "0" if sentence.strip()[0].lower() in "abcdefghijklm" else "1" for sentence in text.split(".") if sentence != ""
    )


# Converts text to a binary string of 0s and 1s, where each word is a 0 if the length of the word is four or less and a 1 if it is five or greater
def decode_length_word_binary(text: str):
    return "".join("0" if len(word) <= 4 else "1" for word in text.split(" ") if word != "")


# Converts text to a binary string of 0s and 1s, where each sentence is a 0 if the length of the first word in the sentence is four or less and a 1 if it is five or greater
def decode_length_sentence_binary(text: str):
    return "".join(
        "0" if len(sentence.strip().split(" ")[0]) <= 4 else "1" for sentence in text.split(".") if sentence != ""
    )


# Converts text to a sequence of letters, where each letter is the first letter of the sentence it is in
def decode_sentence_letter(text: str):
    return "".join(sentence.strip().split(" ")[0][0] for sentence in text.split(".") if sentence != "")


# Converts text to a sequence of letters, where each letter is the first letter of every other sentence, starting with the first one
def decode_alt_sentence_letter(text: str):
    return "".join(sentence.strip().split(" ")[0][0] for i, sentence in enumerate(text.split(".")) if i % 2 == 1)


# Converts text to a sequence of letters, where each letter is the first letter of every third word, starting with the first word
def decode_third_word_letter(text: str):
    words = text.split(" ")
    letters = "".join([words[i][0] for i in range(0, len(words), 3)])
    return letters


# Converts text to a sequence of letters, where each letter is the rot13 of the first letter of every third word, starting with the first word
def decode_rot13_third_word_letter(text: str):
    words = text.split(" ")
    letters = "".join([words[i][0] for i in range(0, len(words), 3)])
    return rotn_encode(letters, 13)


def decode_every_other_letter(text: str):
    return "".join(text[i] for i in range(0, len(text), 2))
