from dataclasses import dataclass
from typing import Any, Dict, List


@dataclass
class BaselineResponse:
    """Single baseline response for a cover task"""

    prompt: str  # The cover task prompt
    response: str
    metadata: Dict[str, Any]  # Store category and other metadata

    def to_dict(self) -> Dict[str, Any]:
        return {"prompt": self.prompt, "response": self.response, "metadata": self.metadata}

    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> "BaselineResponse":
        return cls(prompt=data["prompt"], response=data["response"], metadata=data.get("metadata", {}))


@dataclass
class BaselineSet:
    """Collection of baseline responses for different cover tasks"""

    responses: Dict[str, List[BaselineResponse]]  # cover_task_prompt -> list of responses

    def to_dict(self) -> Dict[str, List[Dict[str, Any]]]:
        """Convert to JSON-serializable dict"""
        return {str(prompt): [r.to_dict() for r in responses] for prompt, responses in self.responses.items()}

    @classmethod
    def from_dict(cls, data: Dict[str, List[Dict[str, Any]]]) -> "BaselineSet":
        """Create from JSON-loaded dict"""
        return cls(
            responses={
                str(prompt): [BaselineResponse.from_dict(r) for r in responses] for prompt, responses in data.items()
            }
        )
