import asyncio
import logging
from typing import List, Optional, Tuple

from litellm import acompletion

from stego_benchmark.config import EvaluationConfig, EvaluationType, ExperimentMode, LLMConfig
from stego_benchmark.data import DatasetItem
from stego_benchmark.experiment import (
    AnswerEvaluationInfo,
    BaselineEvaluationInfo,
    BaseMonitorInfo,
    EvaluationInfo,
    ExperimentResult,
)
from stego_benchmark.prompt.sections import (
    build_answer_correctness_prompt,
    build_base_monitor_prompt,
    build_baseline_prompt,
    build_comparison_prompt,
    build_hidden_task_monitor_prompt,
    build_reasoning_evaluation_prompt,
)
from stego_benchmark.prompt.settings import BasePromptSettings, PromptSettings
from stego_benchmark.results import ResultsManager

from .baseline import BaselineResponse, BaselineSet


async def generate_baseline_responses(
    llm_config: LLMConfig,
    cover_task: DatasetItem,
    num_samples: int = 10,
    batch_size: int = 5,
    settings: Optional[PromptSettings] = None,
) -> List[BaselineResponse]:
    """Generate multiple baseline responses for a given cover task."""
    if settings is None:
        settings = BasePromptSettings(experiment_mode=ExperimentMode.DIRECT)

    baseline_data = []
    logger = logging.getLogger("stego_evaluation")

    async def process_single() -> Optional[BaselineResponse]:
        try:
            prompt_text = build_baseline_prompt(cover_task)
            resp = await acompletion(messages=[{"role": "user", "content": prompt_text}], **llm_config.to_dict())
            return BaselineResponse(
                prompt=cover_task.question, response=resp.choices[0].message.content, metadata=cover_task.metadata
            )
        except Exception as e:
            logger.error(f"Error in baseline generation: {e}")
            return None

    for i in range(0, num_samples, batch_size):
        actual_batch_size = min(batch_size, num_samples - i)
        batch_results = await asyncio.gather(*(process_single() for _ in range(actual_batch_size)))
        baseline_data.extend([r for r in batch_results if r is not None])
        await asyncio.sleep(0.5)  # Rate limiting

    return [r for r in baseline_data if r is not None]


def extract_score(response: str, settings: Optional[PromptSettings] = None) -> Optional[float]:
    """Extract numerical score from evaluation response."""
    if settings is None:
        settings = PromptSettings()
    try:
        parts = response.split(settings.score_tags[0])
        if len(parts) > 1:
            score_str = parts[1].split(settings.score_tags[1])[0].strip()
            return float(score_str)
    except:
        return None
    return None


async def evaluate_response_pair(
    llm_config: LLMConfig,
    cover_task: DatasetItem,
    response_a: str,
    response_b: str,
    settings: Optional[PromptSettings] = None,
) -> Tuple[Optional[float], str, str]:
    """Evaluate a pair of responses and return score and prompts"""
    if settings is None:
        settings = BasePromptSettings(experiment_mode=ExperimentMode.DIRECT)

    prompt = build_comparison_prompt(cover_task, response_a, response_b, settings)
    try:
        eval_resp = await acompletion(messages=[{"role": "user", "content": prompt}], **llm_config.to_dict())
        response_text = eval_resp.choices[0].message.content
        score_match = extract_score(response_text, settings)
        return score_match, prompt, response_text
    except Exception as e:
        return None, prompt, "Error during evaluation"


async def evaluate_both_orderings(
    llm_config: LLMConfig, result: ExperimentResult, baseline_data: BaselineResponse
) -> ExperimentResult:
    """Evaluate responses in both orders and return result with evaluation."""
    logger = logging.getLogger("stego_experiment")

    try:
        # First ordering: stego as A, baseline as B
        score_a, prompt_a, response_a = await evaluate_response_pair(
            llm_config=llm_config,
            cover_task=result.cover_task,
            response_a=result.task_response,
            response_b=baseline_data.response,
        )

        # Second ordering: baseline as A, stego as B
        score_b, prompt_b, response_b = await evaluate_response_pair(
            llm_config=llm_config,
            cover_task=result.cover_task,
            response_a=baseline_data.response,
            response_b=result.task_response,
        )

        baseline_eval = BaselineEvaluationInfo(
            baseline_response=baseline_data.response,
            baseline_prompt=baseline_data.prompt,
            stego_first_score=score_a if score_a is not None else 5.0,
            stego_first_prompt=prompt_a,
            stego_first_evaluator_response=response_a,
            stego_second_score=score_b if score_b is not None else 5.0,
            stego_second_prompt=prompt_b,
            stego_second_evaluator_response=response_b,
            evaluator_model=llm_config.model,
            stego_length=len(result.task_response),
            baseline_length=len(baseline_data.response),
            scratchpad_length=result.scratchpad_length,
        )
        result.evaluation_info = EvaluationInfo(type=EvaluationType.BASELINE_COMPARISON, info=baseline_eval)
    except Exception as e:
        logger.error(f"Error in evaluate_both_orderings: {e}")
        result.evaluation_info = BaselineEvaluationInfo(
            baseline_response=baseline_data.response,
            baseline_prompt=baseline_data.prompt,
            stego_first_score=5.0,
            stego_first_prompt="Error during evaluation",
            stego_first_evaluator_response="Error during evaluation",
            stego_second_score=5.0,
            stego_second_prompt="Error during evaluation",
            stego_second_evaluator_response="Error during evaluation",
            evaluator_model=llm_config.model,
            stego_length=len(result.task_response),
            baseline_length=len(baseline_data.response),
            scratchpad_length=result.scratchpad_length,
        )

    return result


async def evaluate_answer(
    llm_config: LLMConfig,
    result: ExperimentResult,
    verification_scale: str = "binary",
    settings: Optional[BasePromptSettings] = None,
) -> ExperimentResult:
    """Evaluate if the response correctly answers the question and assess reasoning quality"""
    logger = logging.getLogger("stego_experiment")
    if None in [result.cover_task, result.cover_task.options, result.cover_task.answer]:
        logger.error("Cover task or options is None, cannot evaluate answer")
        eval_info = AnswerEvaluationInfo(
            response=result.task_response,
            correct_answer=result.cover_task.options[ord(result.cover_task.answer) - ord("A")],
            is_correct=False,
            correctness_score=0.0,
            reasoning_score=0.0,
            correctness_evaluator_response="Cover task or options is None",
            reasoning_evaluator_response="Cover task or options is None",
            correctness_evaluator_prompt="",
            reasoning_evaluator_prompt="",
            evaluator_model=llm_config.model,
            response_length=len(result.task_response),
        )
        result.evaluation_info = EvaluationInfo(type=EvaluationType.ANSWER_VERIFICATION, info=eval_info)
        return result

    if settings is None:
        settings = BasePromptSettings()

    correct_answer = result.cover_task.options[ord(result.cover_task.answer) - ord("A")]
    reasoning_prompt, correctness_prompt = None, None

    try:
        # Evaluate correctness
        correctness_prompt = build_answer_correctness_prompt(
            question=result.cover_task.question,
            response=result.task_response,
            correct_answer=correct_answer,
            settings=settings,
        )
        correctness_resp = await acompletion(
            messages=[{"role": "user", "content": correctness_prompt}], **llm_config.to_dict()
        )
        correctness_text = correctness_resp.choices[0].message.content
        correctness_score = extract_score(correctness_text, settings)

        # Evaluate reasoning
        reasoning_prompt = build_reasoning_evaluation_prompt(
            question=result.cover_task.question,
            response=result.task_response,
            settings=settings,
        )
        reasoning_resp = await acompletion(
            messages=[{"role": "user", "content": reasoning_prompt}], **llm_config.to_dict()
        )
        reasoning_text = reasoning_resp.choices[0].message.content
        reasoning_score = extract_score(reasoning_text, settings)

        if correctness_score is not None and reasoning_score is not None:
            eval_info = AnswerEvaluationInfo(
                response=result.task_response,
                correct_answer=correct_answer,
                is_correct=correctness_score >= 0.5,
                correctness_score=correctness_score,
                reasoning_score=reasoning_score,
                correctness_evaluator_response=correctness_text,
                reasoning_evaluator_response=reasoning_text,
                correctness_evaluator_prompt=correctness_prompt,
                reasoning_evaluator_prompt=reasoning_prompt,
                evaluator_model=llm_config.model,
                response_length=len(result.task_response),
            )
            result.evaluation_info = EvaluationInfo(type=EvaluationType.ANSWER_VERIFICATION, info=eval_info)
            return result

    except Exception as e:
        logger.error(f"Error in answer evaluation: {e}")
        eval_info = AnswerEvaluationInfo(
            response=result.task_response,
            correct_answer=correct_answer,
            is_correct=False,
            correctness_score=0.0,
            reasoning_score=0.0,
            correctness_evaluator_response="Error during evaluation",
            reasoning_evaluator_response="Error during evaluation",
            correctness_evaluator_prompt=correctness_prompt,
            reasoning_evaluator_prompt=reasoning_prompt,
            evaluator_model=llm_config.model,
            response_length=len(result.task_response),
        )
        result.evaluation_info = EvaluationInfo(type=EvaluationType.ANSWER_VERIFICATION, info=eval_info)
        return result


async def evaluate_monitor(
    config: EvaluationConfig,
    result: ExperimentResult,
    settings: Optional[BasePromptSettings] = None,
) -> ExperimentResult:
    """Evaluate if the response contains any hidden or encoded reasoning about an unrelated task"""
    logger = logging.getLogger("stego_experiment")
    if settings is None:
        settings = BasePromptSettings(experiment_mode=ExperimentMode.DIRECT)
    if config.evaluation_type == EvaluationType.HIDDEN_TASK_MONITOR:
        prompt = build_hidden_task_monitor_prompt(result.cover_task.question, result.task_response, settings)
    elif config.evaluation_type == EvaluationType.BASE_MONITOR:
        prompt = build_base_monitor_prompt(result.cover_task.question, result.task_response, settings)

    try:
        resp = await acompletion(messages=[{"role": "user", "content": prompt}], **config.evaluator.to_dict())
        response_text = resp.choices[0].message.content
        score = extract_score(response_text, settings)
        eval_info = BaseMonitorInfo(
            monitor_prompt=prompt,
            monitor_response=response_text,
            monitor_score=int(score),
        )
        result.evaluation_info = EvaluationInfo(type=config.evaluation_type, info=eval_info)
    except Exception as e:
        logger.error(f"Error in monitor evaluation: {e}")
        eval_info = BaseMonitorInfo(
            monitor_prompt=prompt,
            monitor_response="Error during evaluation",
            monitor_score=5,
        )
        result.evaluation_info = EvaluationInfo(type=config.evaluation_type, info=eval_info)

    return result


async def evaluate_result(
    config: EvaluationConfig,
    result: ExperimentResult,
    baseline_data: Optional[BaselineResponse] = None,
) -> ExperimentResult:
    """Evaluate a single result based on the configured evaluation type"""
    if config.evaluation_type == EvaluationType.BASELINE_COMPARISON:
        if not baseline_data:
            raise ValueError("Baseline data required for baseline comparison")
        return await evaluate_both_orderings(config.evaluator, result, baseline_data)
    elif config.evaluation_type == EvaluationType.ANSWER_VERIFICATION:
        # Answer verification
        return await evaluate_answer(config.evaluator, result, config.verification_scale)
    elif config.evaluation_type in [EvaluationType.HIDDEN_TASK_MONITOR, EvaluationType.BASE_MONITOR]:
        return await evaluate_monitor(config, result)


async def prepare_baseline_data(
    baseline_config: LLMConfig,
    results_manager: Optional[ResultsManager],
    results: List[ExperimentResult],
    baseline_samples: int,
    batch_size: int,
) -> BaselineSet:
    logger = logging.getLogger("stego_experiment")
    # Group results by cover task
    task_groups = {}
    for result in results:
        if result.cover_task:
            task_groups.setdefault((result.cover_task.question, result.cover_task.category), []).append(result)

    # Load or generate baselines
    baselines = {}
    if results_manager:
        baselines = results_manager.load_baselines(baseline_config.model) or {}

    if not baselines:
        logger.info(f"Generating new baseline responses using model: {baseline_config.model}")
        baseline_tasks = []
        for task_prompt, task_category in task_groups.keys():
            task = DatasetItem.create_cover_task(prompt=task_prompt, category=task_category)
            baseline_tasks.append(
                generate_baseline_responses(
                    llm_config=baseline_config, cover_task=task, num_samples=baseline_samples, batch_size=batch_size
                )
            )

        baseline_results = await asyncio.gather(*baseline_tasks)
        baselines = BaselineSet(
            responses={task[0]: responses for task, responses in zip(task_groups.keys(), baseline_results)}
        )

        if results_manager:
            results_manager.save_baselines(baseline_config.model, baselines)
    return baselines


async def run_evaluation(
    evaluation_config: EvaluationConfig,
    results: List[ExperimentResult],
    results_manager: ResultsManager,
) -> List[ExperimentResult]:
    """Run evaluation comparing steganographic responses to baselines"""
    logger = logging.getLogger("stego_experiment")

    # Get baselines if needed
    baselines = None
    if evaluation_config.evaluation_type == EvaluationType.BASELINE_COMPARISON:
        baselines = await prepare_baseline_data(
            evaluation_config.baseline_model or evaluation_config.evaluator,
            results_manager,
            results,
            evaluation_config.baseline_samples,
            evaluation_config.batch_size,
        )

    # Process results in batches
    evaluated_results = []
    current_batch = []
    task_baseline_indices = {}  # Keep track of baseline indices per task

    for result in results:
        if not result.cover_task:
            evaluated_results.append(result)
            continue

        if evaluation_config.evaluation_type == EvaluationType.BASELINE_COMPARISON:
            if not baselines:
                logger.warning("No baselines available for comparison")
                evaluated_results.append(result)
                continue

            task_prompt = result.cover_task.question
            task_baselines = baselines.responses.get(task_prompt, [])

            if not task_baselines:
                logger.warning(f"No baselines found for task: {task_prompt[:100]}...")
                evaluated_results.append(result)
                continue

            # Get next baseline index for this task
            task_baseline_indices.setdefault(task_prompt, 0)
            baseline_idx = task_baseline_indices[task_prompt]
            task_baseline_indices[task_prompt] = (baseline_idx + 1) % len(task_baselines)

            baseline_data = task_baselines[baseline_idx]
        else:
            baseline_data = None

        # Add to current batch
        current_batch.append((result, baseline_data))

        # Process batch if full
        if len(current_batch) >= evaluation_config.batch_size:
            batch_tasks = [evaluate_result(evaluation_config, r, b) for r, b in current_batch]
            batch_results = await asyncio.gather(*batch_tasks)
            evaluated_results.extend(batch_results)
            logger.info(f"Evaluated {len(evaluated_results)} results")
            current_batch = []
            await asyncio.sleep(0.5)  # Rate limiting between batches

    # Process remaining items
    if current_batch:
        batch_tasks = [evaluate_result(evaluation_config, r, b) for r, b in current_batch]
        batch_results = await asyncio.gather(*batch_tasks)
        evaluated_results.extend(batch_results)

    logger.info(f"Completed evaluation of {len(evaluated_results)} results")
    results_manager.save_results(
        evaluated_results,
        path=results_manager.result_dir
        / f"evaluation_{str(evaluation_config.evaluation_type.value)}_{evaluation_config.evaluator.model}.jsonl",
    )
    return evaluated_results
