import json
import logging
import os
import traceback
from typing import Optional

import aiohttp
from litellm import acompletion

# Create a dedicated file handler for litellm errors
litellm_error_logger = logging.getLogger("litellm_errors")
litellm_error_logger.setLevel(logging.ERROR)

# Create a file handler
error_file_handler = logging.FileHandler("litellm_errors.log")
error_file_handler.setLevel(logging.ERROR)

# Create formatter
formatter = logging.Formatter("%(asctime)s - %(levelname)s - %(message)s")
error_file_handler.setFormatter(formatter)

# Add the handler to the logger
litellm_error_logger.addHandler(error_file_handler)

# Keep the standard litellm logger at WARNING level
logging.getLogger("LiteLLM").setLevel(logging.WARNING)


async def unified_completion(
    model: str,
    messages: list,
    timeout: int = 400,
    temperature: float = 1.0,
    max_tokens: Optional[int] = None,
    other_params: Optional[dict] = None,
):
    if model == "deepseek/deepseek-r1":
        # Use aiohttp for deepseek API
        url = "https://openrouter.ai/api/v1/chat/completions"
        headers = {"Authorization": f"Bearer {os.environ['OPENROUTER_API_KEY']}", "Content-Type": "application/json"}
        payload = {"model": model, "messages": messages, "include_reasoning": True, "max_tokens": max_tokens}

        async with aiohttp.ClientSession() as session:
            async with session.post(url, headers=headers, json=payload) as response:
                if response.status == 200:
                    result = await response.json()
                    if "choices" not in result or not result["choices"]:
                        print("Got an unexpected result from DeepSeek:", result)

                    logprobs_out = (
                        result["choices"][0]["logprobs"]["content"][0]["top_logprobs"]
                        if "logprobs" in result["choices"][0]
                        else None
                    )

                    return (
                        result["choices"][0]["message"]["content"],
                        result["choices"][0]["message"]["reasoning"],
                        logprobs_out,
                    )
                else:
                    error_text = await response.text()
                    raise Exception(f"API call failed with status {response.status}: {error_text}")

    else:
        try:
            if other_params is None:
                other_params = {}
            if "vertex_ai" in model:
                with open(os.environ["GOOGLE_APPLICATION_CREDENTIALS"], "r") as f:
                    vertex_credentials = json.load(f)
                other_params["vertex_credentials"] = json.dumps(vertex_credentials)

            response = await acompletion(
                model=model,
                messages=messages,
                timeout=timeout,
                temperature=temperature,
                max_tokens=max_tokens,
                verbose=False,
                **other_params,
            )
            if other_params is not None and other_params.get("top_logprobs") is not None:
                logprobs_out = response.choices[0].logprobs["content"]
            else:
                logprobs_out = None

            return response.choices[0].message.content, None, logprobs_out

        except Exception as e:
            # Log the litellm error
            error_details = {"model": model, "error": str(e), "error_type": type(e).__name__}
            litellm_error_logger.error(f"LiteLLM error with model {model}: {str(e)}")
            litellm_error_logger.error(traceback.format_exc())

            # Re-raise the exception after logging
            raise
