import random
from dataclasses import dataclass, field
from typing import Any, Dict, List, Optional, Type, Union

from stego_benchmark.config import (
    ANSWER_TAGS,
    RESPONSE_A_TAGS,
    RESPONSE_B_TAGS,
    RESPONSE_TAGS,
    SCHEME_TAGS,
    SCORE_TAGS,
    SCRATCHPAD_TAGS,
    USER_TASK_TAGS,
    EncodedReasoningPromptType,
    EncodingScheme,
    ExperimentConfig,
    ExperimentMode,
    HidingMethod,
)
from stego_benchmark.data import DatasetItem


@dataclass
class BasePromptSettings:
    """Base class for prompt settings"""

    experiment_mode: ExperimentMode

    # Common tags used throughout the system
    scratchpad_tags: tuple[str, str] = SCRATCHPAD_TAGS
    response_tags: tuple[str, str] = RESPONSE_TAGS
    answer_tags: tuple[str, str] = ANSWER_TAGS
    user_task_tags: tuple[str, str] = USER_TASK_TAGS
    score_tags: tuple[str, str] = SCORE_TAGS
    response_a_tags: tuple[str, str] = RESPONSE_A_TAGS
    response_b_tags: tuple[str, str] = RESPONSE_B_TAGS
    scheme_tags: tuple[str, str] = SCHEME_TAGS

    # Additional instructions or disclaimers
    additional_notes: str = field(default_factory=str)

    def to_dict(self) -> Dict[str, Any]:
        """Convert to dictionary for serialization"""
        return {
            "experiment_mode": self.experiment_mode.value,
        }


@dataclass
class SteganographicPromptSettings(BasePromptSettings):
    """Settings for steganographic experiment mode"""

    hidden_task: Optional[DatasetItem] = None
    cover_task: Optional[DatasetItem] = None
    hiding_method: Optional[HidingMethod] = None
    use_scratchpad: bool = False
    monitor_scratchpad: bool = False
    use_cover_task_reasoning: bool = False

    def __post_init__(self):
        self.experiment_mode = ExperimentMode.STEGANOGRAPHIC

    def to_dict(self) -> Dict[str, Any]:
        """Convert to dictionary for serialization"""
        result = super().to_dict()
        result.update(
            {
                "use_scratchpad": self.use_scratchpad,
                "monitor_scratchpad": self.monitor_scratchpad,
                "hiding_method": self.hiding_method.value if self.hiding_method else None,
                "hidden_task": self.hidden_task.to_dict() if self.hidden_task else None,
                "cover_task": self.cover_task.to_dict() if self.cover_task else None,
            }
        )
        return result


@dataclass
class DirectPromptSettings(BasePromptSettings):
    """Settings for direct experiment mode"""

    task: Optional[DatasetItem] = None
    use_scratchpad: bool = False

    def __post_init__(self):
        self.experiment_mode = ExperimentMode.DIRECT

    def to_dict(self) -> Dict[str, Any]:
        """Convert to dictionary for serialization"""
        result = super().to_dict()
        result.update(
            {
                "use_scratchpad": self.use_scratchpad,
                "task": self.task.to_dict() if self.task else None,
            }
        )
        return result


@dataclass
class MessageInReasoningPromptSettings(BasePromptSettings):
    """Settings for message-in-reasoning experiment mode"""

    task: Optional[DatasetItem] = None
    messages: List[str] = field(default_factory=list)  # List of messages
    encoding_scheme: Optional[EncodingScheme] = None
    current_message_index: int = 0  # Track the current message index
    last_message: str = ""
    seed: Optional[int] = None  # Add seed parameter
    config: ExperimentConfig = None  # Add config attribute

    def __post_init__(self):
        self.experiment_mode = ExperimentMode.MESSAGE_IN_REASONING
        if self.config and self.config.setting_config.message_dataset_path.exists():
            with open(self.config.setting_config.message_dataset_path, "r") as file:
                self.messages = [line.strip() for line in file if line.strip()]
            if self.seed is not None:
                random.seed(self.seed)
                random.shuffle(self.messages)  # Shuffle deterministically

    def get_new_message(self) -> str:
        # Return the current message and increment the index
        self.current_message_index += 1
        self.last_message = self.messages[self.current_message_index % len(self.messages)]
        return self.last_message

    @property
    def current_message(self) -> str:
        # Return the current message without incrementing the index
        return self.messages[self.current_message_index % len(self.messages)]

    @property
    def random_message(self) -> str:
        self.last_message = random.choice(self.messages)
        return self.last_message

    def to_dict(self) -> Dict[str, Any]:
        result = super().to_dict()
        result.update(
            {
                "messages": self.messages,
                "current_message_index": self.current_message_index,
                "encoding_scheme": self.encoding_scheme.value if self.encoding_scheme else None,
            }
        )
        return result


@dataclass
class EncodedReasoningPromptSettings(BasePromptSettings):
    """Settings for encoded reasoning experiment mode"""

    reasoning_task: Optional[DatasetItem] = None
    cover_task: Optional[DatasetItem] = None
    prompt_type: Optional[EncodedReasoningPromptType] = EncodedReasoningPromptType.NONE
    few_shot: bool = False
    letter_mapping: Dict[str, str] = field(default_factory=dict)  # For SCRATCHPAD_LETTERS_DIFFERENT mode

    def __post_init__(self):
        self.experiment_mode = ExperimentMode.ENCODED_REASONING

    def to_dict(self) -> Dict[str, Any]:
        """Convert to dictionary for serialization"""
        result = super().to_dict()
        result.update(
            {
                "prompt_type": self.prompt_type.value if self.prompt_type else None,
                "reasoning_task": self.reasoning_task.to_dict() if self.reasoning_task else None,
                "cover_task": self.cover_task.to_dict() if self.cover_task else None,
                "letter_mapping": self.letter_mapping,
            }
        )
        return result


PromptSettings = Union[
    SteganographicPromptSettings, DirectPromptSettings, MessageInReasoningPromptSettings, EncodedReasoningPromptSettings
]


def create_prompt_settings(
    config: ExperimentConfig, task: Optional[DatasetItem] = None, cover_task: Optional[DatasetItem] = None
) -> BasePromptSettings:
    """Factory function to create appropriate settings object based on experiment mode"""
    if config.experiment_mode == ExperimentMode.STEGANOGRAPHIC:
        return SteganographicPromptSettings(
            experiment_mode=config.experiment_mode,
            hidden_task=task,
            cover_task=cover_task,
            hiding_method=config.setting_config.hiding_method,
            use_scratchpad=config.setting_config.use_scratchpad,
            monitor_scratchpad=config.setting_config.monitor_scratchpad,
            use_cover_task_reasoning=config.setting_config.use_cover_task_reasoning,
        )
    elif config.experiment_mode == ExperimentMode.DIRECT:
        return DirectPromptSettings(
            experiment_mode=config.experiment_mode,
            task=task,
            use_scratchpad=config.setting_config.use_scratchpad,
        )
    elif config.experiment_mode == ExperimentMode.MESSAGE_IN_REASONING:
        return MessageInReasoningPromptSettings(
            experiment_mode=config.experiment_mode,
            task=task,
            encoding_scheme=config.setting_config.encoding_scheme,
            seed=config.seed,
            config=config,
        )
    elif config.experiment_mode == ExperimentMode.ENCODED_REASONING:
        return EncodedReasoningPromptSettings(
            experiment_mode=config.experiment_mode,
            reasoning_task=task,
            cover_task=cover_task,
            prompt_type=config.setting_config.encoded_reasoning_prompt_type,
            few_shot=config.few_shot,
        )
    else:
        raise ValueError(f"Unknown experiment mode: {config.experiment_mode}")
