import datetime
import json
import logging
import os
import re
import uuid
from dataclasses import asdict
from pathlib import Path
from typing import List, Optional, Union

import yaml

from stego_benchmark.config import (
    BenchmarkConfig,
    DirectConfig,
    ExperimentMode,
    HidingMethod,
    MessageInReasoningConfig,
    SteganographicConfig,
)
from stego_benchmark.evaluation.baseline import BaselineSet
from stego_benchmark.experiment import (
    DirectExperimentResult,
    EncodedReasoningExperimentResult,
    ExperimentResult,
    MessageInReasoningExperimentResult,
    SteganographicExperimentResult,
)


def generate_results_filename(
    model_name: str,
    hiding_method: HidingMethod,
    use_scratchpad: bool,
    num_samples: Optional[int] = None,
    experiment_mode: ExperimentMode = ExperimentMode.STEGANOGRAPHIC,
) -> str:
    """Generate a unique filename for experiment results based on parameters."""
    base_name = f"{model_name}_{experiment_mode.value}"
    base_name = re.sub(r"[^a-zA-Z0-9_]", "_", base_name)  # sanitize
    if experiment_mode == ExperimentMode.STEGANOGRAPHIC:
        base_name += f"_{hiding_method.value}"
    if use_scratchpad:
        base_name += "_scratchpad"
    if num_samples:
        base_name += f"_{num_samples}samples"

    # Just return the filename without path
    return f"{base_name}.json"


class ResultsManager:
    """Manages saving and loading experiment results"""

    def __init__(self, config: BenchmarkConfig):
        self.config = config
        # Create result directory using timestamp
        self.results_dir = config.experiment.output_dir
        self.results_dir.mkdir(parents=True, exist_ok=True)
        self.results_file = None

    def init_results_file(self) -> None:
        """Create an empty results file"""
        timestamp = datetime.datetime.now().strftime("%Y%m%d_%H%M%S")
        experiment_mode = self.config.experiment.experiment_mode.value
        if experiment_mode == ExperimentMode.STEGANOGRAPHIC.value:
            if isinstance(self.config.experiment.setting_config, SteganographicConfig):
                method_name = self.config.experiment.setting_config.hiding_method.value
                self.result_dir = (
                    Path(self.config.experiment.output_dir) / f"{timestamp}_{experiment_mode}_{method_name}"
                )
            else:
                self.result_dir = Path(self.config.experiment.output_dir) / f"{timestamp}_{experiment_mode}"
        elif experiment_mode == ExperimentMode.MESSAGE_IN_REASONING.value:
            if isinstance(self.config.experiment.setting_config, MessageInReasoningConfig):
                encoding_scheme = self.config.experiment.setting_config.encoding_scheme.value
                self.result_dir = (
                    Path(self.config.experiment.output_dir) / f"{timestamp}_{experiment_mode}_{encoding_scheme}"
                )
            else:
                self.result_dir = Path(self.config.experiment.output_dir) / f"{timestamp}_{experiment_mode}"
        elif experiment_mode == ExperimentMode.SIMPLE_ENCODE_OR_DECODE.value:
            encoding_scheme = self.config.experiment.setting_config.encoding_scheme.value
            model_name = self.config.experiment.model.model.split("/")[-1]
            cot_str = "cot" if self.config.experiment.setting_config.include_cot else "nocot"
            forward_str = "forward" if self.config.experiment.setting_config.forward_bool else "backward"
            self.result_dir = (
                Path(self.config.experiment.output_dir)
                / f"{timestamp}_{cot_str}_{forward_str}_{encoding_scheme}_{model_name}"
            )
        elif experiment_mode == ExperimentMode.ENCODE_AND_DECODE.value:
            model_name = self.config.experiment.model.model.split("/")[-1]
            encoding_scheme = self.config.experiment.setting_config.encoding_scheme.value
            cot_str = "cot" if self.config.experiment.setting_config.include_cot else "nocot"
            self.result_dir = (
                Path(self.config.experiment.output_dir) / f"{timestamp}_{cot_str}_{encoding_scheme}_{model_name}"
            )
        elif experiment_mode == ExperimentMode.ENCODE_BASELINE.value:
            model_name = self.config.experiment.model.model.split("/")[-1]
            self.result_dir = Path(self.config.experiment.output_dir) / f"{timestamp}_{model_name}"
        else:
            # For direct mode or fallback
            self.result_dir = Path(self.config.experiment.output_dir) / f"{timestamp}_{experiment_mode}"

        self.result_dir.mkdir(parents=True, exist_ok=True)

        # Save config
        self.config.save(self.result_dir / "config.yaml")

        # Create results JSON file
        self.results_file = self.result_dir / "results.jsonl"
        self.evaluation_file = self.result_dir / "evaluation.jsonl"
        if not self.results_file.exists():
            with open(self.results_file, "w") as f:
                f.write("")  # Create empty file

    def save_results(
        self,
        results: List[Union[ExperimentResult]],
        path: Optional[Path] = None,
    ) -> None:
        """Save a batch of results to JSONL file"""
        if self.results_file is None:
            self.init_results_file()
        if path is None:
            path = self.results_file
        with open(path, "w", encoding="utf-8") as f:
            results_list = [result.to_dict() for result in results]
            json.dump(results_list, f, indent=2)

    def get_results_path(self) -> Path:
        """Return path to results file"""
        return self.results_file

    def get_baseline_path(self, model_name: str) -> Path:
        """Get path for baseline results"""
        return self.result_dir / f"baselines_{model_name.replace('/', '_')}.json"

    def save_baselines(self, model_name: str, baselines: BaselineSet) -> None:
        """Save baseline results"""
        baseline_file = self.get_baseline_path(model_name)
        if not baseline_file.exists():
            baseline_file.parent.mkdir(parents=True, exist_ok=True)
        with open(baseline_file, "w", encoding="utf-8") as f:
            json.dump(baselines.to_dict(), f, indent=2, ensure_ascii=False)

    def load_baselines(self, model_name: str) -> Optional[BaselineSet]:
        """Load baseline results if they exist"""
        baseline_file = self.get_baseline_path(model_name)
        if not baseline_file.exists():
            return None
        try:
            with open(baseline_file, "r", encoding="utf-8") as f:
                data = json.load(f)
                return BaselineSet.from_dict(data)
        except json.JSONDecodeError as e:
            logger = logging.getLogger("stego_experiment")
            logger.error(f"Error loading baselines: {e}")
            return None

    def load_results(self, path: Optional[Union[str, Path]] = None) -> List[ExperimentResult]:
        """Load results from JSON file"""
        if isinstance(path, str):
            path = Path(path)
        if path is None:
            path = self.results_file
        # check if filepath is file or directory
        print(path)
        if path.is_dir():
            # check if there is a results.jsonl file in the directory
            results_file = path / "results.jsonl"
            if not results_file.exists():
                # check if there is another directory in the directory
                for subdir in path.iterdir():
                    print(subdir)
                    if subdir.is_dir():
                        results_file = subdir / "results.jsonl"
                        if results_file.exists():
                            path = results_file
                            self.results_file = results_file
                            self.result_dir = path.parent
                            break
        if not path.exists():
            return []
        with open(path, "r", encoding="utf-8") as f:
            data = json.load(f)
        exp_mode = ExperimentMode(self.config.experiment.experiment_mode)
        if exp_mode == ExperimentMode.STEGANOGRAPHIC:
            return [SteganographicExperimentResult.from_dict(d) for d in data]
        elif exp_mode == ExperimentMode.MESSAGE_IN_REASONING:
            return [MessageInReasoningExperimentResult.from_dict(d) for d in data]
        elif exp_mode == ExperimentMode.DIRECT:
            return [DirectExperimentResult.from_dict(d) for d in data]
        else:
            return [EncodedReasoningExperimentResult.from_dict(d) for d in data]
