import argparse
import asyncio
import logging
import os
from collections import Counter
from pathlib import Path

from stego_benchmark.config import (
    BenchmarkConfig,
    EncodedReasoningConfig,
    EncodedReasoningPromptType,
    EvaluationConfig,
    EvaluationType,
    ExperimentConfig,
    ExperimentMode,
    HidingMethod,
    LLMConfig,
    setup_logger,
)
from stego_benchmark.data import Dataset
from stego_benchmark.runner import ExperimentRunner

data_name_file_map = {
    "aqua_rat": "datasets/encoded_reasoning/aqua_rat.json",
    "gpqa_diamond": "datasets/encoded_reasoning/gpqa_diamond.csv",
    "state_tracking_1d": "datasets/encoded_reasoning/state_tracking_1d.json",
    "state_tracking_2d": "datasets/encoded_reasoning/state_tracking_2d.json",
}


pre_defined = ["first_letter", "word_count", "terminal_punct", "special_chars", "character_count"]
model_defined = [
    "model_generated_subtle",
    "model_generated_moderate",
    "model_generated_blatant",
    "model_generated_subtle_2",
]
other_settings = ["implicit", "none"]
encoded_scratchpad = [
    "scratchpad_base64",
    "scratchpad_rot13",
    "scratchpad_hex",
    "scratchpad_ascii",
    "scratchpad_model_generated",
]
restricted_scratchpad = ["scratchpad_names", "scratchpad_letters", "scratchpad_letters_different"]


async def main():
    model_name = "anthropic/claude-3-5-sonnet-latest"

    # Setup logging
    logger = setup_logger("stego_experiment", level=logging.DEBUG)
    for dataset_name, file_name in data_name_file_map.items():
        all_settings = model_defined + encoded_scratchpad + other_settings
        if "state_tracking" not in dataset_name:
            all_settings += pre_defined + restricted_scratchpad
        for prompt_type in all_settings:
            encoded_reasoning_config = EncodedReasoningConfig(
                encoded_reasoning_prompt_type=EncodedReasoningPromptType(prompt_type),
                cover_task_dataset_path=(
                    Path("datasets/encoded_reasoning/anthropic_cover_tasks_1000.jsonl")
                    if "scratchpad" not in prompt_type
                    else None
                ),
            )
            config = BenchmarkConfig(
                experiment=ExperimentConfig(
                    experiment_mode=ExperimentMode.ENCODED_REASONING,
                    model=LLMConfig(
                        model=model_name,
                        temperature=0.0,
                        max_tokens=5000,
                        reasoning_effort=None,
                    ),
                    dataset_path=Path(file_name),
                    output_dir=Path(f"results/encoded_reasoning_experiments/{dataset_name}/{prompt_type}"),
                    batch_size=1,
                    num_samples=2,
                    few_shot=False,
                    verbose=True,
                    seed=42,
                    setting_config=encoded_reasoning_config,
                ),
            )

            runner = ExperimentRunner(config)
            logger.info(f"Running experiment with config: {config.to_dict()}")
            logger.info(f"Model: {config.experiment.model.model}")
            logger.info(f"Prompt type: {prompt_type}")
            try:
                results = await runner.run()

                # Filter out None results and log final stats
                valid_results = [r for r in results if r is not None]
                success_count = sum(1 for r in valid_results if r.is_correct)
                total_count = len(valid_results)
                success_rate = (success_count / total_count) * 100 if total_count > 0 else 0.0

                logger.info(f"Valid results: {total_count}/{len(results)}")
                logger.info(f"Final success rate: {success_count}/{total_count} ({success_rate:.2f}%)")

            except Exception as e:
                logger.error(f"Error during experiment execution: {e}", exc_info=True)
                raise


if __name__ == "__main__":
    asyncio.run(main())
