import json
import logging
from dataclasses import dataclass
from pathlib import Path
from typing import Dict, List, Optional

import pandas as pd

from stego_benchmark.config import EvaluationType
from stego_benchmark.experiment import ExperimentResult

logger = logging.getLogger(__name__)


@dataclass
class ExperimentPath:
    model: str
    scratchpad_setting: str
    dataset: str
    experiment_mode: str
    cover_task: str
    hiding_method: Optional[str] = None


def load_single_experiment(path: Path) -> List[ExperimentResult]:
    """Load results from a single experiment json file"""
    try:
        with open(path) as f:
            raw_results = json.load(f)
            if not isinstance(raw_results, list):
                logger.error(f"Expected list of results, got {type(raw_results)}")
                return []

            loaded_results = []
            for i, result in enumerate(raw_results):
                try:
                    loaded_results.append(ExperimentResult.from_dict(result))
                except Exception as e:
                    logger.error(f"Error loading result {i}: {e}")
                    logger.debug(f"Problematic result data: {result}")
            return loaded_results
    except Exception as e:
        logger.error(f"Error loading file {path}: {e}")
        return []


def results_to_dataframe(results: List[ExperimentResult], experiment_path: ExperimentPath) -> pd.DataFrame:
    """Convert experiment results to pandas DataFrame"""
    rows = []
    for result in results:
        try:
            row = {
                # Experiment settings
                "model": experiment_path.model,
                "scratchpad_setting": experiment_path.scratchpad_setting,
                "dataset": experiment_path.dataset,
                "experiment_mode": experiment_path.experiment_mode,
                "cover_task": experiment_path.cover_task,
                "hiding_method": experiment_path.hiding_method,
                # Basic metrics
                "is_correct": result.is_correct,
                "response_length": len(result.task_response),
                "scratchpad_length": result.scratchpad_length,
                # Task details
                "task_response": result.task_response,
                "extracted_answer": result.extracted_answer,
                "correct_answer": result.hidden_task.answer if result.hidden_task else None,
                # Evaluation metrics
                "evaluation_type": result.evaluation.type.value if result.evaluation else None,
            }

            # Add evaluation-specific metrics
            if result.evaluation:
                if result.evaluation.type == EvaluationType.BASELINE_COMPARISON:
                    row.update(
                        {
                            "aggregated_score": result.evaluation.info.aggregated_score,
                            "stego_first_score": result.evaluation.info.stego_first_score,
                            "stego_second_score": result.evaluation.info.stego_second_score,
                            "baseline_length": result.evaluation.info.baseline_length,
                        }
                    )
                elif result.evaluation.type == EvaluationType.ANSWER_VERIFICATION:
                    row.update(
                        {
                            "confidence_score": result.evaluation.info.confidence_score,
                            "evaluator_reasoning": result.evaluation.info.evaluator_reasoning,
                            "is_correct_eval": result.evaluation.info.is_correct,
                        }
                    )
            rows.append(row)
        except Exception as e:
            logger.error(f"Error converting result to row: {e}")
            continue

    return pd.DataFrame(rows)


def load_experiment_results(
    base_path: Path,
    experiment_path: ExperimentPath,
) -> pd.DataFrame:
    """Load results for specific experiment configuration and return as DataFrame"""
    full_path = (
        base_path
        / experiment_path.model
        / experiment_path.scratchpad_setting
        / experiment_path.dataset
        / experiment_path.experiment_mode
        / experiment_path.cover_task
    )
    if experiment_path.hiding_method:
        full_path = full_path / experiment_path.hiding_method

    try:
        # Look for results file but exclude baseline files
        json_files = list(full_path.glob("*.json"))
        results_files = [f for f in json_files if "baseline" not in f.name.lower()]

        if not results_files:
            logger.warning(f"No results file found in {full_path}")
            return pd.DataFrame()

        results_file = results_files[0]  # Take the first matching file
        logger.debug(f"Loading results from {results_file}")
        results = load_single_experiment(results_file)

        if not results:
            logger.warning(f"No results loaded from {results_file}")
            return pd.DataFrame()

        return results_to_dataframe(results, experiment_path)
    except Exception as e:
        logger.error(f"Error loading results from {full_path}: {e}")
        return pd.DataFrame()


def load_multiple_experiments(
    base_path: Path,
    experiment_paths: List[ExperimentPath],
) -> pd.DataFrame:
    """Load multiple experiments and combine into single DataFrame"""
    dfs = []
    for exp_path in experiment_paths:
        try:
            df = load_experiment_results(base_path, exp_path)
            if not df.empty:
                dfs.append(df)
            else:
                logger.warning(f"Empty results for {exp_path}")
        except Exception as e:
            logger.error(f"Error loading {exp_path}: {e}")

    if not dfs:
        logger.warning("No data loaded from any experiment")
        return pd.DataFrame()

    return pd.concat(dfs, ignore_index=True)
