from pathlib import Path
from typing import Dict, List, Optional

import matplotlib.pyplot as plt
import numpy as np
import pandas as pd
import seaborn as sns

from stego_benchmark.config import EvaluationType
from stego_benchmark.experiment import ExperimentResult

from .metrics import calculate_success_rate


def plot_success_rates_by_method(
    results_by_method: Dict[str, List[ExperimentResult]],
    title: str,
    output_path: Path,
) -> None:
    """Plot success rates for different hiding methods"""
    methods = []
    rates = []

    for method, results in results_by_method.items():
        methods.append(method)
        rates.append(calculate_success_rate(results))

    plt.figure(figsize=(10, 6))
    sns.barplot(x=methods, y=rates)
    plt.title(title)
    plt.ylabel("Success Rate (%)")
    plt.xticks(rotation=45)
    plt.tight_layout()
    plt.savefig(output_path)
    plt.close()


def plot_length_distribution(
    results: List[ExperimentResult],
    title: str,
    output_path: Path,
) -> None:
    """Plot distribution of response lengths"""
    lengths = [len(r.task_response) for r in results]

    plt.figure(figsize=(10, 6))
    sns.histplot(lengths, bins=30)
    plt.title(title)
    plt.xlabel("Response Length")
    plt.ylabel("Count")
    plt.tight_layout()
    plt.savefig(output_path)
    plt.close()


def plot_evaluation_scores_comparison(
    baseline_results: List[ExperimentResult],
    stego_results: List[ExperimentResult],
    title: str,
    output_path: Path,
) -> None:
    """Plot comparison of evaluation scores between baseline and stego responses"""
    baseline_scores = []
    stego_scores = []

    for r in baseline_results:
        if r.evaluation and r.evaluation.type == EvaluationType.BASELINE_COMPARISON:
            baseline_scores.append(r.evaluation.info.aggregated_score)

    for r in stego_results:
        if r.evaluation and r.evaluation.type == EvaluationType.BASELINE_COMPARISON:
            stego_scores.append(r.evaluation.info.aggregated_score)

    plt.figure(figsize=(10, 6))
    sns.boxplot(data=[baseline_scores, stego_scores], labels=["Baseline", "Steganographic"])
    plt.title(title)
    plt.ylabel("Evaluation Score")
    plt.tight_layout()
    plt.savefig(output_path)
    plt.close()


def plot_success_rate_heatmap(
    results_by_dataset_and_method: Dict[str, Dict[str, List[ExperimentResult]]],
    title: str,
    output_path: Path,
) -> None:
    """Create heatmap of success rates across datasets and methods"""
    data = {}
    for dataset, method_results in results_by_dataset_and_method.items():
        data[dataset] = {method: calculate_success_rate(results) for method, results in method_results.items()}

    df = pd.DataFrame(data)

    plt.figure(figsize=(12, 8))
    sns.heatmap(df, annot=True, fmt=".1f", cmap="YlOrRd")
    plt.title(title)
    plt.tight_layout()
    plt.savefig(output_path)
    plt.close()


def plot_grouped_bars(
    df: pd.DataFrame,
    x_column: str,
    group_column: str,
    y_column: str = "is_correct",
    count_mode: bool = False,
    title: str = None,
    output_path: Optional[Path] = None,
    figsize: tuple = (12, 6),
    y_label: Optional[str] = None,
    rotation: int = 45,
    additional_filters: Optional[dict] = None,
    include_none: bool = False,
    max_groups: int = 10,
    ax: Optional[plt.Axes] = None,
    color_mapping: Optional[Dict[str, str]] = None,
    style: str = "seaborn-v0_8-paper",
) -> None:
    """
    Create a grouped bar plot from DataFrame.

    Args:
        df: DataFrame containing the data
        x_column: Column to use for x-axis categories
        group_column: Column to use for grouping bars
        y_column: Column to calculate means for (default: 'is_correct')
        count_mode: If True, plot counts instead of means (ignores y_column)
        title: Plot title
        output_path: Path to save the plot
        figsize: Figure size as (width, height)
        y_label: Label for y-axis (defaults to "Count" in count mode)
        rotation: Rotation of x-axis labels
        additional_filters: Dict of {column: value} to filter the DataFrame
        include_none: If True, include None values in the plot, if False, filter them out
        max_groups: Maximum number of groups to show (will take top N by frequency)
        ax: Optional Axes object to plot on
        color_mapping: Dictionary mapping group values to colors
        style: Matplotlib style to apply
    """
    # Clear the current figure if not using provided axis
    if ax is None:
        plt.clf()

    # Set the style
    with plt.style.context(style):
        # Apply additional filters if provided
        plot_df = df.copy()
        if additional_filters:
            for col, val in additional_filters.items():
                if isinstance(val, list):
                    plot_df = plot_df[plot_df[col].isin(val)]
                else:
                    plot_df = plot_df[plot_df[col] == val]

        # Handle None values
        if not include_none:
            plot_df = plot_df[plot_df[group_column].notna()]
            plot_df = plot_df[plot_df[x_column].notna()]

        if plot_df.empty:
            print(f"No data to plot after filtering")
            return

        # Print unique values count for debugging
        print(f"Number of unique values in {group_column}: {plot_df[group_column].nunique()}")
        print(f"Number of unique values in {x_column}: {plot_df[x_column].nunique()}")

        # If too many groups, take only the top N most frequent ones
        if plot_df[group_column].nunique() > max_groups:
            top_groups = plot_df[group_column].value_counts().nlargest(max_groups).index
            plot_df = plot_df[plot_df[group_column].isin(top_groups)]
            print(f"Limiting to top {max_groups} most frequent groups in {group_column}")

        # Calculate statistics
        if count_mode:
            # Create all possible combinations of x and group values
            x_vals = sorted([x for x in plot_df[x_column].unique() if x is not None])
            if include_none and plot_df[x_column].isna().any():
                x_vals = [None] + x_vals

            group_vals = sorted([g for g in plot_df[group_column].unique() if g is not None])
            if include_none and plot_df[group_column].isna().any():
                group_vals = [None] + group_vals

            # Create index for all combinations
            index = pd.MultiIndex.from_product([x_vals, group_vals], names=[x_column, group_column])

            # Calculate counts and reindex to include all combinations
            counts = plot_df.groupby([x_column, group_column]).size()
            counts = counts.reindex(index, fill_value=0).reset_index(name="count")

            summary_df = counts
            summary_df["mean"] = summary_df["count"]
            summary_df["se"] = 0
            if y_label is None:
                y_label = "Count"
        else:
            if y_column == "is_correct" or y_column == "is_correct_eval":
                # For binary data, use binomial proportion confidence interval
                x_vals = sorted([x for x in plot_df[x_column].unique() if x is not None])
                if include_none and plot_df[x_column].isna().any():
                    x_vals = [None] + x_vals

                group_vals = sorted([g for g in plot_df[group_column].unique() if g is not None])
                if include_none and plot_df[group_column].isna().any():
                    group_vals = [None] + group_vals

                # Create index for all combinations
                index = pd.MultiIndex.from_product([x_vals, group_vals], names=[x_column, group_column])

                # Calculate statistics and reindex
                # Convert to numeric and handle NaN values
                plot_df[y_column] = pd.to_numeric(plot_df[y_column], errors="coerce")
                plot_df[y_column] = plot_df[y_column].fillna(0)  # Replace NaN with 0

                stats = plot_df.groupby([x_column, group_column])[y_column].agg(["mean", "count"])
                stats = stats.reindex(index, fill_value=0).reset_index()

                summary_df = stats
                summary_df["mean"] = summary_df["mean"] * 100

                # Calculate standard error avoiding NaN and inf values
                means = summary_df["mean"].to_numpy() / 100
                counts = np.maximum(summary_df["count"].to_numpy(), 1)  # Avoid division by zero

                # Calculate variance safely
                variance = np.clip(means * (1 - means), 0, 1)  # Ensure values are between 0 and 1

                # Calculate standard error
                summary_df["se"] = 100 * np.sqrt(variance / counts)

                if y_label is None:
                    y_label = "Success Rate (%)"
            else:
                # Similar approach for non-binary data
                # ... (implement if needed)
                pass

        # Use provided axis or create new figure
        if ax is None:
            fig = plt.figure(figsize=figsize, dpi=100)
            ax = fig.add_subplot(111)

        n_groups = len(group_vals)
        if n_groups == 0:
            print(f"No groups to plot")
            return

        # Set bar width and positions
        width = 0.8 / n_groups
        positions = np.arange(len(x_vals))

        # Plot bars for each group
        for i, group in enumerate(group_vals):
            group_data = summary_df[summary_df[group_column] == group]
            offset = (i - (n_groups - 1) / 2) * width

            # Use color from mapping if provided, otherwise use seaborn color palette
            if color_mapping:
                color = color_mapping.get(str(group))
            else:
                # Use seaborn's color palette
                colors = sns.color_palette("husl", n_groups)
                color = colors[i]

            ax.bar(
                positions + offset,
                group_data["mean"],
                width,
                label=str(group),
                yerr=group_data["se"] if not count_mode else None,
                capsize=3,  # Smaller cap size
                color=color,
                alpha=0.8,  # Slight transparency
                edgecolor="white",  # White edges
                linewidth=1,  # Thin edges
            )

        # Customize the plot
        ax.set_xlabel(x_column.replace("_", " ").title(), fontsize=12, labelpad=10)
        ax.set_ylabel(y_label, fontsize=12, labelpad=10)
        ax.set_ylim(0, 105)  # Slightly higher to avoid cutting off error bars

        if title:
            ax.set_title(title, fontsize=12, pad=20)

        # Customize grid
        ax.yaxis.grid(True, linestyle="--", alpha=0.7, zorder=0)
        ax.set_axisbelow(True)  # Put grid below bars

        # Remove top and right spines
        ax.spines["top"].set_visible(False)
        ax.spines["right"].set_visible(False)

        # Customize ticks
        ax.set_xticks(positions)
        ax.set_xticklabels(x_vals, rotation=rotation)
        ax.tick_params(axis="both", labelsize=9)

        # Customize legend
        ax.legend(
            title=group_column.replace("_", " ").title(),
            title_fontsize=15,
            fontsize=12,
            frameon=True,
            framealpha=0.9,
            edgecolor="none",
            bbox_to_anchor=(1.05, 1),
            loc="upper left",
            borderaxespad=0,
            markerscale=1.5,
        )

        # Adjust layout to make room for the bigger legend
        plt.tight_layout()
        plt.subplots_adjust(right=0.85)

        # Add value labels inside bars
        for i, group in enumerate(group_vals):
            group_data = summary_df[summary_df[group_column] == group]
            offset = (i - (n_groups - 1) / 2) * width
            for x, mean in zip(positions, group_data["mean"]):
                y_pos = mean / 2
                ax.text(
                    x + offset,
                    y_pos,
                    f"{mean:.0f}" if count_mode else f"{mean:.1f}",
                    ha="center",
                    va="center",
                    fontsize=8,
                    bbox=dict(
                        facecolor="white",
                        alpha=0.7,
                        edgecolor="none",
                        pad=1.5,
                        boxstyle="round,pad=0.5",  # Rounded corners
                    ),
                )

        if output_path:
            plt.savefig(output_path, bbox_inches="tight", dpi=300)
            plt.close()
        elif ax is None:  # Only show if this is a standalone plot
            plt.show()
            plt.close()  # Close the figure after showing


def plot_grouped_boxplots(
    df: pd.DataFrame,
    x_column: str,
    group_column: str,
    y_column: str,
    title: str = None,
    output_path: Optional[Path] = None,
    figsize: tuple = (12, 6),
    y_label: Optional[str] = None,
    rotation: int = 45,
    additional_filters: Optional[dict] = None,
    include_none: bool = False,
    max_groups: int = 10,
    show_points: bool = True,
    ax: Optional[plt.Axes] = None,
    color_mapping: Optional[Dict[str, str]] = None,
    style: str = "seaborn-v0_8-paper",
    ylim: Optional[int] = None,
) -> None:
    """
    Create a grouped box plot from DataFrame.

    Args:
        df: DataFrame containing the data
        x_column: Column to use for x-axis categories
        group_column: Column to use for grouping boxes
        y_column: Column containing numerical values to plot
        title: Plot title
        output_path: Path to save the plot
        figsize: Figure size as (width, height)
        y_label: Label for y-axis
        rotation: Rotation of x-axis labels
        additional_filters: Dict of {column: value} to filter the DataFrame
        include_none: If True, include None values in the plot
        max_groups: Maximum number of groups to show
        show_points: Whether to show individual points
        ax: Optional Axes object to plot on
        color_mapping: Dictionary mapping group values to colors
        style: Matplotlib style to apply
    """
    # Clear the current figure if not using provided axis
    if ax is None:
        plt.clf()

    # Set the style
    with plt.style.context(style):
        # Use provided axis or create new figure
        if ax is None:
            fig = plt.figure(figsize=figsize, dpi=100)
            ax = fig.add_subplot(111)

        # Apply additional filters if provided
        plot_df = df.copy()
        if additional_filters:
            for col, val in additional_filters.items():
                if isinstance(val, list):
                    plot_df = plot_df[plot_df[col].isin(val)]
                else:
                    plot_df = plot_df[plot_df[col] == val]

        # Handle None values
        if not include_none:
            plot_df = plot_df[plot_df[group_column].notna()]
            plot_df = plot_df[plot_df[x_column].notna()]
            plot_df = plot_df[plot_df[y_column].notna()]

        if plot_df.empty:
            print(f"No data to plot after filtering")
            return

        # Print unique values count for debugging
        print(f"Number of unique values in {group_column}: {plot_df[group_column].nunique()}")
        print(f"Number of unique values in {x_column}: {plot_df[x_column].nunique()}")

        # If too many groups, take only the top N most frequent ones
        if plot_df[group_column].nunique() > max_groups:
            top_groups = plot_df[group_column].value_counts().nlargest(max_groups).index
            plot_df = plot_df[plot_df[group_column].isin(top_groups)]
            print(f"Limiting to top {max_groups} most frequent groups in {group_column}")

        # Create box plot with custom colors if provided
        palette = color_mapping if color_mapping else "husl"
        sns.boxplot(
            data=plot_df,
            x=x_column,
            y=y_column,
            hue=group_column,
            width=0.8,
            zorder=2,
            order=sorted(plot_df[x_column].unique()),
            hue_order=sorted(plot_df[group_column].unique()),
            palette=palette,
            ax=ax,
            linewidth=1,
            fliersize=3,
        )

        if show_points:
            # Add individual points behind boxes
            sns.stripplot(
                data=plot_df,
                x=x_column,
                y=y_column,
                hue=group_column,
                dodge=True,
                size=3,
                alpha=0.3,
                zorder=1,
                legend=False,
                ax=ax,
                palette=palette,
            )

        # Customize the plot
        ax.set_xlabel(x_column.replace("_", " ").title(), fontsize=12, labelpad=10)
        ax.set_ylabel(y_label if y_label else y_column.replace("_", " ").title(), fontsize=12, labelpad=10)
        if ylim:
            ax.set_ylim(0, ylim)

        if title:
            ax.set_title(title, fontsize=12, pad=20)

        # Customize grid
        ax.yaxis.grid(True, linestyle="--", alpha=0.7, zorder=0)
        ax.set_axisbelow(True)  # Put grid below boxes

        # Remove top and right spines
        ax.spines["top"].set_visible(False)
        ax.spines["right"].set_visible(False)

        # Customize ticks
        ax.tick_params(axis="both", labelsize=9)
        plt.xticks(rotation=rotation)

        # Customize legend
        ax.legend(
            title=group_column.replace("_", " ").title(),
            title_fontsize=15,
            fontsize=12,
            frameon=True,
            framealpha=0.9,
            edgecolor="none",
            bbox_to_anchor=(1.05, 1),
            loc="upper left",
            borderaxespad=0,
            markerscale=1.5,
        )

        # Adjust layout to make room for the bigger legend
        plt.tight_layout()
        plt.subplots_adjust(right=0.85)

        # Add statistical annotations if needed
        # ... (optional: add mean values or other statistics)

        if output_path:
            plt.savefig(output_path, bbox_inches="tight", dpi=300)
            plt.close()
        elif ax is None:  # Only show if this is a standalone plot
            plt.show()
            plt.close()  # Close the figure after showing


def plot_grid(
    df: pd.DataFrame,
    plot_func: callable,
    plot_kwargs: dict,
    row_field: Optional[str] = None,
    col_field: Optional[str] = None,
    figsize: tuple = (16, 10),
    title: Optional[str] = None,
    output_path: Optional[Path] = None,
    spacing: tuple = (0.1, 0.2),
) -> None:
    """
    Create a grid of plots with rows and columns defined by unique values in specified fields.
    At least one of row_field or col_field must be specified.

    Args:
        df: DataFrame containing the data
        row_field: Column name to use for rows (optional)
        col_field: Column name to use for columns (optional)
        plot_func: Function to create individual plots (plot_grouped_bars or plot_grouped_boxplots)
        plot_kwargs: Dictionary of arguments to pass to plot_func
        figsize: Figure size for the entire grid
        title: Main title for the entire figure
        output_path: Path to save the plot
        spacing: Tuple of (horizontal, vertical) spacing between subplots

    Raises:
        ValueError: If neither row_field nor col_field is specified
    """
    if row_field is None and col_field is None:
        raise ValueError("At least one of row_field or col_field must be specified")

    # Get unique values for rows and columns
    if row_field:
        row_vals = sorted(df[row_field].unique())
        n_rows = len(row_vals)
    else:
        row_vals = [None]
        n_rows = 1

    if col_field:
        col_vals = sorted(df[col_field].unique())
        n_cols = len(col_vals)
    else:
        col_vals = [None]
        n_cols = 1

    # Adjust figure size based on grid layout
    if row_field and not col_field:
        # Vertical layout
        figsize = (figsize[0] * 0.8, figsize[1] * n_rows / 2)
    elif col_field and not row_field:
        # Horizontal layout
        figsize = (figsize[0] * n_cols / 2, figsize[1] * 0.8)

    # Create figure and grid of subplots with higher DPI
    fig, axes = plt.subplots(
        n_rows,
        n_cols,
        figsize=figsize,
        squeeze=False,
        dpi=300,  # Increased DPI for better quality
        constrained_layout=True,  # Better layout management
    )

    # Adjust spacing between subplots and make room for title
    plt.subplots_adjust(
        hspace=spacing[1],
        wspace=spacing[0],
        top=0.82,  # Keep this value to maintain space for title
        bottom=0.1,
        left=0.1,
        right=0.9,
    )

    for i, row_val in enumerate(row_vals):
        for j, col_val in enumerate(col_vals):
            # Filter data for this subplot
            plot_data = df.copy()
            if row_field:
                plot_data = plot_data[plot_data[row_field] == row_val]
            if col_field:
                plot_data = plot_data[plot_data[col_field] == col_val]

            # Update plot_kwargs
            current_kwargs = plot_kwargs.copy()
            current_kwargs["df"] = plot_data
            current_kwargs["ax"] = axes[i, j]
            current_kwargs["output_path"] = None

            # Add subplot title
            subplot_title = []
            if row_field and row_val is not None:
                subplot_title.append(f"{row_field}: {row_val}")
            if col_field and col_val is not None:
                subplot_title.append(f"{col_field}: {col_val}")
            current_kwargs["title"] = "\n".join(subplot_title)

            # Create plot
            with plt.rc_context({"axes.titlesize": 10}):
                plot_func(**current_kwargs)

            # Only show x labels for bottom row
            if i != n_rows - 1:
                axes[i, j].set_xlabel("")
                axes[i, j].set_xticklabels([])

            # Only show y labels for leftmost column
            if j != 0:
                axes[i, j].set_ylabel("")
                axes[i, j].set_yticklabels([])

            # Only show legend for top-right plot
            if i != 0 or j != n_cols - 1:
                if axes[i, j].get_legend() is not None:
                    axes[i, j].get_legend().remove()

    if title:
        fig.suptitle(
            title,
            fontsize=16,
            y=1.0,  # Moved title higher up (increased from 0.92)
            # fontweight='bold'
        )

    plt.tight_layout()

    if output_path:
        plt.savefig(
            output_path,
            bbox_inches="tight",
            dpi=300,  # Ensure high DPI when saving
            format="png",  # Explicitly specify format
            facecolor="white",  # Ensure white background
            edgecolor="none",
            pad_inches=0.1,  # Add small padding
            transparent=False,  # Ensure non-transparent background
        )
        plt.close()
    else:
        plt.show()


def plot_stacked_percentage_bars(
    df: pd.DataFrame,
    x_column: str,
    group_column: str,
    y_column: str,
    title: str = None,
    output_path: Optional[Path] = None,
    figsize: tuple = (12, 6),
    y_label: Optional[str] = None,
    rotation: int = 45,
    additional_filters: Optional[dict] = None,
    include_none: bool = False,
    color_mapping: Optional[Dict[str, str]] = None,
    ax: Optional[plt.Axes] = None,
    style: str = "seaborn-v0_8-paper",
) -> None:
    """
    Create a stacked percentage bar plot from DataFrame.

    Args:
        df: DataFrame containing the data
        x_column: Column to use for x-axis categories
        group_column: Column to use for grouping bars
        y_column: Column containing categorical values to calculate percentages for
        title: Plot title
        output_path: Path to save the plot
        figsize: Figure size as (width, height)
        y_label: Label for y-axis
        rotation: Rotation of x-axis labels
        additional_filters: Dict of {column: value} to filter the DataFrame
        include_none: If True, include None values
        color_mapping: Dictionary mapping y-values to colors
        ax: Optional Axes object to plot on
        style: Matplotlib style to apply
    """
    # Clear the current figure if not using provided axis
    if ax is None:
        plt.clf()

    # Set the style
    with plt.style.context(style):
        # Use provided axis or create new figure
        if ax is None:
            fig = plt.figure(figsize=figsize, dpi=100)
            ax = fig.add_subplot(111)

        # Apply additional filters if provided
        plot_df = df.copy()
        if additional_filters:
            for col, val in additional_filters.items():
                plot_df = plot_df[plot_df[col] == val]

        # Calculate percentages for each combination
        percentages = (
            plot_df.groupby([x_column, group_column, y_column])
            .size()
            .unstack(fill_value=0)
            .groupby(level=1)
            .apply(lambda x: (x / x.sum() * 100))
        )

        # Get unique values for plotting
        y_vals = sorted(plot_df[y_column].unique())
        group_vals = sorted(plot_df[group_column].unique())
        x_vals = sorted(plot_df[x_column].unique())

        # Set bar width and positions
        n_groups = len(group_vals)
        width = 0.8 / n_groups
        positions = np.arange(len(x_vals))

        # Plot stacked bars for each group
        for i, group in enumerate(group_vals):
            bottom = np.zeros(len(x_vals))
            offset = (i - (n_groups - 1) / 2) * width

            for y_val in y_vals:
                # Get color from mapping or use default colormap
                if color_mapping:
                    color = color_mapping.get(str(y_val))
                else:
                    colors = sns.color_palette("husl", len(y_vals))
                    color = colors[y_vals.index(y_val)]

                # Get heights for this y_value
                heights = percentages.loc[group, y_val]

                # Plot this segment
                ax.bar(
                    positions + offset,
                    heights,
                    width,
                    bottom=bottom,
                    label=f"{group}, {y_val}" if i == 0 else None,  # Only label first group
                    color=color,
                    alpha=0.8,
                    edgecolor="white",
                    linewidth=1,
                )

                # Add percentage labels if significant
                for x, h in enumerate(heights):
                    if h > 5:  # Only show label if segment is >5%
                        ax.text(
                            x + offset,
                            bottom + h / 2,
                            f"{h:.1f}%",
                            ha="center",
                            va="center",
                            fontsize=8,
                            color="white",
                            fontweight="bold",
                        )

                bottom += heights

        # Customize the plot
        ax.set_xlabel(x_column.replace("_", " ").title(), fontsize=12, labelpad=10)
        ax.set_ylabel(y_label if y_label else "Percentage (%)", fontsize=12, labelpad=10)
        ax.set_ylim(0, 100)

        if title:
            ax.set_title(title, fontsize=12, pad=20)

        # Remove top and right spines
        ax.spines["top"].set_visible(False)
        ax.spines["right"].set_visible(False)

        # Customize ticks
        ax.set_xticks(positions)
        ax.set_xticklabels(x_vals, rotation=rotation)
        ax.tick_params(axis="both", labelsize=9)

        # Customize legend
        ax.legend(
            title=f"{group_column} & {y_column}",
            title_fontsize=12,
            fontsize=10,
            frameon=True,
            framealpha=0.9,
            edgecolor="none",
            bbox_to_anchor=(1.05, 1),
            loc="upper left",
            borderaxespad=0,
        )

        # Adjust layout
        plt.tight_layout()
        plt.subplots_adjust(right=0.85)

        if output_path:
            plt.savefig(output_path, bbox_inches="tight", dpi=300)
            plt.close()
        elif ax is None:
            plt.show()
            plt.close()


# Example usage:
"""
# Load all results into a single DataFrame
experiment_paths = [
    ExperimentPath(
        model="claude35sonnet",
        scratchpad_setting=setting,
        dataset=dataset,
        experiment_mode="steganographic",
        cover_task="default",
        hiding_method=method
    )
    for setting in ["no_scratchpad", "with_scratchpad", "monitored_scratchpad"]
    for dataset in ["commonsenseqa", "gpqa_diamond"]
    for method in ["explicit_choice", "first_letter", "word_count"]
]

df = load_multiple_experiments(Path("full_results"), experiment_paths)

# Create plot comparing hiding methods across scratchpad settings
plot_grouped_bars(
    df,
    x_column="hiding_method",
    group_column="scratchpad_setting",
    title="Success Rate by Hiding Method and Scratchpad Setting",
    output_path=Path("plots/hiding_methods_comparison.png"),
    additional_filters={"dataset": "commonsenseqa"}
)

# Create plot comparing datasets across hiding methods
plot_grouped_bars(
    df,
    x_column="dataset",
    group_column="hiding_method",
    title="Success Rate by Dataset and Hiding Method",
    output_path=Path("plots/dataset_comparison.png"),
    additional_filters={"scratchpad_setting": "with_scratchpad"}
)
"""
