import argparse
import asyncio
import json
import logging
import os
from pathlib import Path
from typing import List

from stego_benchmark.config import EvaluationConfig, setup_logger
from stego_benchmark.evaluation.logic import run_evaluation
from stego_benchmark.experiment import ExperimentResult


def setup_logging(verbose: bool = False) -> logging.Logger:
    logger = logging.getLogger("stego_evaluation")
    level = logging.DEBUG if verbose else logging.INFO
    logger.setLevel(level)

    handler = logging.StreamHandler()
    handler.setFormatter(logging.Formatter("%(asctime)s - %(levelname)s - %(message)s"))
    logger.addHandler(handler)

    return logger


async def main():
    parser = argparse.ArgumentParser(description="Evaluate steganographic responses")
    parser.add_argument("--config", type=Path, required=True, help="Path to YAML config file")

    args = parser.parse_args()

    config = EvaluationConfig.from_yaml(args.config)

    logger = setup_logger("stego_experiment", level=logging.DEBUG if config.experiment.verbose else logging.INFO)

    # Load results
    logger.info(f"Loading results from {config.experiment.results_file}")
    with open(config.experiment.results_file, "r", encoding="utf-8") as f:
        data = json.load(f)
    results = [ExperimentResult.from_dict(d) for d in data]

    # Run evaluation
    updated_results = await run_evaluation(
        model_name=args.model, results=results, baseline_samples=args.baseline_samples, batch_size=args.batch_size
    )

    # Save results
    os.makedirs(args.output_dir, exist_ok=True)
    output_path = Path(args.output_dir) / f"evaluated_{Path(args.results_file).name}"

    logger.info(f"Saving evaluated results to {output_path}")
    with open(output_path, "w", encoding="utf-8") as f:
        json.dump([r.to_dict() for r in updated_results], f, indent=2)


if __name__ == "__main__":
    asyncio.run(main())
