import argparse
import asyncio
import logging
from pathlib import Path

from stego_benchmark.config import BenchmarkConfig, EvaluationType, ExperimentMode, HidingMethod, setup_logger
from stego_benchmark.data import Dataset
from stego_benchmark.runner import ExperimentRunner


async def main():
    # Parse command line arguments
    parser = argparse.ArgumentParser(description="Run steganographic experiments")
    parser.add_argument("--config", type=Path, required=True, help="Path to YAML config file")
    args = parser.parse_args()

    # Load configuration
    config = BenchmarkConfig.from_yaml(args.config)

    # Setup logging
    logger = setup_logger("stego_experiment", level=logging.DEBUG if config.experiment.verbose else logging.INFO)

    # Create and run experiment
    runner = ExperimentRunner(config)
    logger.info(f"Running experiment with config: {config.to_dict()}")
    try:
        results = await runner.run()

        # Filter out None results and log final stats
        valid_results = [r for r in results if r is not None]
        success_count = sum(1 for r in valid_results if r.is_correct)
        total_count = len(valid_results)
        success_rate = (success_count / total_count) * 100 if total_count > 0 else 0.0

        logger.info(f"Valid results: {total_count}/{len(results)}")
        logger.info(f"Final success rate: {success_count}/{total_count} ({success_rate:.2f}%)")

    except Exception as e:
        logger.error(f"Error during experiment execution: {e}", exc_info=True)
        raise


if __name__ == "__main__":
    asyncio.run(main())
