import random
from typing import List, Optional

from stego_benchmark.config import BenchmarkConfig, ExperimentMode, setup_logger
from stego_benchmark.data import Dataset
from stego_benchmark.evaluation.logic import run_evaluation
from stego_benchmark.experiment import ExperimentResult, run_experiment_async
from stego_benchmark.results import ResultsManager


class ExperimentRunner:
    """Handles experiment execution and coordination"""

    def __init__(self, config: BenchmarkConfig):
        self.config = config
        random.seed(config.experiment.seed)
        self.logger = setup_logger("stego_experiment")
        self.results_manager = ResultsManager(config)
        # load dataset
        self.dataset = Dataset.load(config.experiment.dataset_path)
        if config.experiment.num_samples:
            self.dataset = self.dataset.sample(config.experiment.num_samples)
        self.cover_tasks = None

        # Handle cover tasks for both steganographic and encoded reasoning modes
        if config.experiment.experiment_mode in [ExperimentMode.STEGANOGRAPHIC, ExperimentMode.ENCODED_REASONING]:
            # Load cover tasks
            if config.experiment.setting_config.cover_task_dataset_path:
                self.cover_tasks = Dataset.load(config.experiment.setting_config.cover_task_dataset_path)
            else:
                self.cover_tasks = None

            # Sample cover tasks if needed
            if config.experiment.setting_config.num_samples_cover_tasks:
                self.cover_tasks = self.cover_tasks.sample(config.experiment.setting_config.num_samples_cover_tasks)
        elif config.experiment.experiment_mode in [ExperimentMode.ENCODE_AND_DECODE, ExperimentMode.ENCODE_BASELINE]:
            task_dataset_path = config.experiment.setting_config.task_dataset_path
            if task_dataset_path is None:
                self.cover_tasks = None
            else:
                self.cover_tasks = Dataset.load(task_dataset_path)

    async def run(self) -> List[ExperimentResult]:
        # Run main experiment
        results = await run_experiment_async(
            config=self.config.experiment,
            dataset=self.dataset,
            results_manager=self.results_manager,
            cover_tasks=self.cover_tasks,
        )

        # Run evaluation if configured
        if self.config.evaluation:
            evaluation_results = await run_evaluation(
                evaluation_config=self.config.evaluation,
                results=results,
                results_manager=self.results_manager,
            )
            # Save evaluated results
            self.results_manager.save_results(evaluation_results)
            return evaluation_results

        return results

    async def run_evaluation(self) -> List[ExperimentResult]:
        return await run_evaluation(
            evaluation_config=self.config.evaluation,
            results=self.results_manager.load_results(path=self.config.experiment.output_dir),
            results_manager=self.results_manager,
        )
