# GeoNeRF is a generalizable NeRF model that renders novel views
# without requiring per-scene optimization. This software is the 
# implementation of the paper "GeoNeRF: Generalizing NeRF with 
# Geometry Priors" by Mohammad Mahdi Johari, Yann Lepoittevin,
# and Francois Fleuret.

# Copyright (c) 2022 ams International AG

# This file is part of GeoNeRF.
# GeoNeRF is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 3 as
# published by the Free Software Foundation.

# GeoNeRF is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with GeoNeRF. If not, see <http://www.gnu.org/licenses/>.

import configargparse

def config_parser():
    parser = configargparse.ArgumentParser()
    parser.add_argument("--config", is_config_file=True, help="Config file path")

    # Datasets options
    parser.add_argument("--dataset_name", type=str, default="llff", choices=["train", "train_new", "llff", "nerf", "dtu", "lf_data","ithaca","timeLapse","tt","photoTourism","waymo_ref"],)
    parser.add_argument("--llff_path", type=str, help="Path to llff dataset")
    parser.add_argument("--llff_test_path", type=str, help="Path to llff dataset")
    parser.add_argument("--dtu_path", type=str, help="Path to dtu dataset")
    parser.add_argument("--dtu_pre_path", type=str, help="Path to preprocessed dtu dataset")
    parser.add_argument("--nerf_path", type=str, help="Path to nerf dataset")
    parser.add_argument("--ams_path", type=str, help="Path to ams dataset")
    parser.add_argument("--ibrnet1_path", type=str, help="Path to ibrnet1 dataset")
    parser.add_argument("--ibrnet2_path", type=str, help="Path to ibrnet2 dataset")
    parser.add_argument("--lf_path", type=str, help="Path to lf dataset")
    parser.add_argument("--ithaca_path", type=str, help="Path to ithaca dataset")
    parser.add_argument("--ithaca_label_path", type=str, help="Path to ithaca label dataset")
    parser.add_argument("--timeLapse_path", type=str, help="Path to timeLapse dataset")
    parser.add_argument("--tt_path", type=str, help="Path to tanksAndTemple dataset")
    parser.add_argument("--photoTourism_path", type=str, help="Path to photoTourism dataset")
    parser.add_argument("--downsample", type=float, default=1.0, help="image down sample")
    parser.add_argument("--llffdownsample", type=float, default=1.0, help="LLFF image down sample")
    parser.add_argument("--only_dtu", action="store_true", help="train on DTU only")
    parser.add_argument("--only_llff", action="store_true", help="train on LLFF only")
    parser.add_argument("--only_ithaca", action="store_true", help="train on ithaca only")
    parser.add_argument("--test_far_view", action="store_true", help="use far view")
    parser.add_argument("--ithaca_use_two_cams", action="store_true", help="use cam0,1,2 instead of forward driving capture as different views")
    parser.add_argument("--src_specify", type=str, default="all", help="specify which weather (source img)")
    parser.add_argument("--ref_specify", type=str, default="all", help="specify which weather (ref img)")
    parser.add_argument("--cam_diff_weather", type=str, default="None", help="test different weather, cam file")
    parser.add_argument("--read_lidar", action="store_true", help="read lidar")
    parser.add_argument("--camfile", type=str, default="colmap", help="ithaca camera file")
    parser.add_argument("--style_dataset", type=str, default="ithaca", help="style dataset")
    parser.add_argument("--specify_file", type=str, default="None", help="specify which data")
    parser.add_argument("--n_output_views", type=int, default=1, help="number of novel view")
    parser.add_argument("--n_split", type=int, default=16, help="number of spliting 2pi")
    parser.add_argument("--pretrain_dataset", type=str, default="ithaca", help="drit++ pretrain style dataset")
    parser.add_argument("--save_video_frames", action="store_true", help="save_video_frames (timeLapse dataset)")
    parser.add_argument("--to_calculate_consistency", action="store_true", help="to calculate consistency (tt dataset)")
    parser.add_argument("--far_consistency", action="store_true", help="to calculate consistency for far views (tt dataset)")
    parser.add_argument("--styleSame", action="store_true", help="style image use input images(ithaca)")
    parser.add_argument("--to_calculate_FID", action="store_true", help="to calculate FID")

    # Training options
    parser.add_argument("--geonerfMDMM", action="store_true", help="geonerf + MDMM")
    parser.add_argument("--ckpt", type=str, default="None", help="checkpoint")
    parser.add_argument("--batch_size", type=int, default=512)
    parser.add_argument("--num_steps", type=int, default=200000)
    parser.add_argument("--n_epochs", type=int, default=10)
    parser.add_argument("--nb_views", type=int, default=3)
    parser.add_argument("--lrate", type=float, default=5e-4, help="Learning rate")
    parser.add_argument("--warmup_steps", type=int, default=500, help="Gradually warm-up learning rate in optimizer")
    parser.add_argument("--scene", type=str, default="None", help="Scene for fine-tuning")
    parser.add_argument("--disocclude", action="store_true", help="disocclude confidence")
    parser.add_argument("--upperbound", action="store_true", help="disocclude confidence use ground truth to construct")
    parser.add_argument("--upperbound_noise", action="store_true", help="GT depth +noise")
    parser.add_argument("--upperbound_gauss", action="store_true", help="GT depth; D construct using gaussian cdf")
    parser.add_argument("--D_gauss", action="store_true", help="D construct using gaussian cdf")
    parser.add_argument("--Deither0or1", action="store_true", help="disocclude confidence either 0 or 1")
    # parser.add_argument("--geoFeatComplete", action="store_true", help="geo feataure completion (occupancy info.)")
    parser.add_argument("--geoFeatComplete", type=str, default="None", help="geo feataure completion (occupancy info.)")
    parser.add_argument("--texFeat", action="store_true", help="add texture feataure (texture info.)")
    parser.add_argument("--texFeatComplete", action="store_true", help="texture feataure completion (texture info.)")
    parser.add_argument("--renderer_v1", action="store_true", help="renderer geometry part modify")
    parser.add_argument("--renderer_v1_mlp", action="store_true", help="renderer geometry part modify (mlp, no MHA)")
    parser.add_argument("--renderer_v2", action="store_true", help="renderer appearance part modify")
    parser.add_argument("--renderer_density_v1", action="store_true", help="renderer geometry part modify")
    parser.add_argument("--renderer_RGBresidual", action="store_true", help="renderer RGB part modify: c + sigma(w*(c-img))")
    parser.add_argument("--renderer_geneRGBsigma", action="store_true", help="renderer generate RGB, density for mask_sum=0")
    parser.add_argument("--renderer_geneRGBsigma_dist", action="store_true", help="renderer generate RGB, density distributions for mask_sum=0")
    parser.add_argument("--renderer_mvslike", action="store_true", help="renderer architecture similar to mvsnerf")
    parser.add_argument("--renderer_mvslike_v2", action="store_true", help="renderer architecture similar to mvsnerf+geonerf")
    parser.add_argument("--weightedMeanVar", action="store_true", help="input of renderer: weighted mean&var with disocc confi")
    parser.add_argument("--attention_3d", action="store_true", help="add attention 3d feature")
    parser.add_argument("--check_feat_mode", type=str, default="None", help="check feat depend on cost volume(0:use only level2; 1:use another network to generate feature(size=level2); 2:surface:ori feat, occluded: new feat)")
    parser.add_argument("--load_costreg", action="store_true", help="load costreg")
    parser.add_argument("--separate_occ_feat", action="store_true", help="seperate occ feature (v_feat_A, v_feat_B)(same as check_feat_mode 2, but using level 0-2)")
    parser.add_argument("--update_casmvs", action="store_true", help="update casmvs, freeze remaining")
    parser.add_argument("--learn_3dfeat_from_GT", action="store_true", help="v_feat(casmvs)(from GT) <-> v_feat(from pred)")
    parser.add_argument("--use_midas", action="store_true", help="use midas")
    parser.add_argument("--train_patch", action="store_true", help="train patch") # unfinish!
    parser.add_argument("--cas_confi", action="store_true", help="casmvs depth confi")
    parser.add_argument("--texFeat_woUnet", action="store_true", help="add texture feataure (directly concat img)")
    parser.add_argument("--feat3dMask", action="store_true", help="feature 3d range mask (outside of the range:0 ; else:1)")
    parser.add_argument("--use_angle_cos", action="store_true", help="direction difference using cos value (instead of turn into angle then positional encoding)")
    parser.add_argument("--which_level", type=str, default="2", help="since renderer_mvslike would output dict(contains 3 levels), choose which level to output result")
    parser.add_argument("--stage1", action="store_true", help="stage 1 : loss on input views")
    parser.add_argument("--train_vis_novel", action="store_true", help="stage 1 : loss on visible part of novel view")
    parser.add_argument("--tex_2Dto3D", action="store_true", help=" weight:(related to geometry feat O(indicate surface) and direction diff), texture feature in 3d: weighted 2d feature map/image color")
    parser.add_argument("--use_tex_bias", action="store_true", help=" weight:(related to geometry feat O(indicate surface) and direction diff), texture feature in 3d: weighted 2d feature map/image color")
    parser.add_argument("--use_angle_both_cos_PE", action="store_true", help="direction difference using cos value & positional encoding")
    parser.add_argument("--weighted_rgb", action="store_true", help="rgb=weight*img_color")
    parser.add_argument("--catDomain", action="store_true", help="concat domain vec for every levels (rendererStyle)")
    parser.add_argument("--catStyle", action="store_true", help="concat style for every generate_layers (rendererStyle)")
    parser.add_argument("--styleMLP_cls", action="store_true", help="different classes have different style MLP")
    parser.add_argument("--styleLast", action="store_true", help="content adain (all levels) then style adain")
    parser.add_argument("--NVSThenStyle", action="store_true", help="(1) train NVS (2) train style img (fix params except style-related network)")
    parser.add_argument("--adainNormalize", action="store_true", help="adaIn x normalize")
    parser.add_argument("--style3Dfeat", action="store_true", help="style 3D feature")
    parser.add_argument("--timephi", action="store_true", help="style -> time (f(phi)) (discard weather)")
    parser.add_argument("--freezeExceptTimephi", action="store_true", help="freeze params except time phi network")
    parser.add_argument("--styleTwoBranch", action="store_true", help="renderer style 2 branches (common content/special content)")
    parser.add_argument("--weatherEmbedding", action="store_true", help="weather embedding")
    parser.add_argument("--weatherEncode", action="store_true", help="weather encode")
    parser.add_argument("--weatherEncodeCls", action="store_true", help="weather encode and predict encoding class (no night class)")
    parser.add_argument("--styleTwoBranch_ver", type=str, default="v0", help="styleTwoBranch version")
    parser.add_argument("--wo_z", action="store_true", help="styleTwoBranch branch2 no input z")
    parser.add_argument("--add_z", action="store_true", help="RendererStyle input z")
    parser.add_argument("--phi_final_actv", type=str, default="lrelu", help="phi2code_network final_actv (2 branches)")
    parser.add_argument("--use_fourier_feature", action="store_true", help="z_cos_code & z_sin_code use_fourier_feature (2 branches)")
    parser.add_argument("--gauss_var", type=float, default=1.0, help="random z~(0,var)")
    parser.add_argument("--branch2_noPhi", action="store_true", help="branch2_noPhi")
    parser.add_argument("--zInputStyle", action="store_true", help="train: z is the style of ref image; test: z is the style of input image")
    parser.add_argument("--unparallExtract", action="store_true", help="z is the style of input image")
    parser.add_argument("--z_dim", type=int, default=8, help="z dim")
    parser.add_argument("--fourier_phi", action="store_true", help="positional encoding(phi)")
    parser.add_argument("--phi_noCosSin", action="store_true", help="phi no cos,sin")
    parser.add_argument("--noLoadCas", action="store_true", help="no load CasMVS")
    parser.add_argument("--noFreezeContent", action="store_true", help="no freeze content extractor")
    parser.add_argument("--contentPyramid", action="store_true", help="content feature use pyramid network")
    parser.add_argument("--update_z", action="store_true", help="update z freeze everything else")
    parser.add_argument("--contentFeature", action="store_true", help="use content feature")
    parser.add_argument("--delta_t", action="store_true", help="delta t estimator + delta_t loss")
    parser.add_argument("--delta_t_1x1", action="store_true", help="delta t estimator is 1x1 convs")
    parser.add_argument("--z_zInputStyle_Fuse", action="store_true", help="style input: (1) z~N(0,1) (2) z=the style of input image")
    parser.add_argument("--nightNoRemainCode", action="store_true", help="if ref img=night image, not to extract remaining code (instead, use input remaining code)")
    parser.add_argument("--zInputStyle_isInput", action="store_true", help="train/test: z is the style of input image")
    parser.add_argument("--phi_reg", action="store_true", help="rgb -> cos(phi),sin(phi)")
    
    # loss
    parser.add_argument("--wo_depth_gt_supervise", action="store_true", help="wo depth gt supervision")
    parser.add_argument("--DT_loss", action="store_true", help="loss: MSE(disocc confi, T)")
    parser.add_argument("--DT_loss_lamb", type=float, default=0.01, help="lambda of DT_loss")
    parser.add_argument("--O_constraint", action="store_true", help="loss: MSE(O(from close view), O(from not so close view))")
    parser.add_argument("--O_loss_lamb", type=float, default=0.001, help="lambda of DT_loss")
    parser.add_argument("--consistent3d_loss", action="store_true", help="loss: 3d consistency (includes RGB, density, T, depth)")
    parser.add_argument("--consist3d_g_loss_lamb", type=float, default=0.0001, help="lambda of consistent3d_loss (geo)")
    parser.add_argument("--consist3d_rgb_loss_lamb", type=float, default=0.1, help="lambda of consistent3d_loss (rgb)")
    parser.add_argument("--consist3d_coord_loss_lamb", type=float, default=10, help="lambda of consistent3d_loss (coord)")
    parser.add_argument("--pDensity", action="store_true", help="casMVS p is density!")
    parser.add_argument("--pDensity_loss", action="store_true", help="loss: MSE(p, density)")
    parser.add_argument("--pDensity_loss_lamb", type=float, default=0.001, help="lambda of pDensity_loss")
    parser.add_argument("--pDensity_RGBloss", action="store_true", help="loss: MSE(2d_rgb(p), 2d_rgb(density))")
    parser.add_argument("--pDensity_RGBloss_lamb", type=float, default=100, help="lambda of pDensity_RGBloss")
    parser.add_argument("--O_label_loss", action="store_true", help="loss: MSE(conv(O), 0 or 1) (only for those ahead of depth)")
    parser.add_argument("--O_label_loss_lamb", type=float, default=0.1, help="lambda of O_label_loss")
    parser.add_argument("--O_label_01loss", action="store_true", help="loss: close to 0/1")
    parser.add_argument("--O_label_01loss_lamb", type=float, default=0.0001, help="lambda of O_label_01loss")
    parser.add_argument("--P_constraint", action="store_true", help="loss: p_l(lower than pred depth):x, p_h(higher than pred depth):(1-x), else:0; pred depth=x*depth_l + (1-x)*depth_h")
    parser.add_argument("--P_loss_lamb", type=float, default=0.1, help="lambda of P_loss")
    parser.add_argument("--unimvs_loss", action="store_true", help="loss: uniMVS")
    parser.add_argument("--unimvs_loss_lamb", type=float, default=1e-5, help="lambda of unimvs_loss")
    parser.add_argument("--density_01loss", action="store_true", help="loss: close to 0/1")
    parser.add_argument("--density_01loss_lamb", type=float, default=0.0001, help="lambda of density_01loss")
    parser.add_argument("--far_view_loss", action="store_true", help="loss: far view loss")
    parser.add_argument("--far_view_loss_lamb", type=float, default=0.5, help="lambda of far_view_loss")
    parser.add_argument("--cycle_loss", action="store_true", help="loss: cycle loss")
    parser.add_argument("--cycle_loss_lamb", type=float, default=1.0, help="lambda of cycle_loss")
    parser.add_argument("--novel_depth_loss", action="store_true", help="loss: novel depth loss")
    parser.add_argument("--edge_loss", action="store_true", help="loss: edge loss")
    parser.add_argument("--edge_loss_weight", type=float, default=0.3, help="weight of edge loss")
    parser.add_argument("--uct_lamb", type=float, default=0.0005, help="lambda of uncertainty loss")
    parser.add_argument("--density_sum_lamb", type=float, default=0.01, help="lambda of density_sum")
    parser.add_argument("--renderer_geneRGBsigma_loss", action="store_true", help="calculate loss for renderer_generate RGB, density for mask_sum=0")
    parser.add_argument("--extra_depth_loss", action="store_true", help="L_ssim, L_smooth, L_DA") # refer to RC-MVS
    parser.add_argument("--L_ssim_lamb", type=float, default=0.2, help="lambda of L_ssim")
    parser.add_argument("--L_smooth_lamb", type=float, default=0.0067, help="lambda of L_smooth")
    parser.add_argument("--L_DA_lamb", type=float, default=0.01, help="lambda of L_DA")
    parser.add_argument("--style_lamb", type=float, default=0.5, help="lambda of style image")
    parser.add_argument("--adaptive_style_loss", action="store_true", help="style loss weight=e^(-loss_domain_cls)")
    parser.add_argument("--vgg_loss", action="store_true", help="patch vgg loss")
    parser.add_argument("--vgg_loss_style", action="store_true", help="patch vgg loss (style_img)")
    parser.add_argument("--vgg_loss_lamb", type=float, default=0.001, help="patch vgg loss lambda")
    parser.add_argument("--contrast_loss", action="store_true", help="contrastive loss")
    parser.add_argument("--contrast_loss_lamb", type=float, default=0.001, help="contrastive loss lambda")
    parser.add_argument("--patch_cs_loss", action="store_true", help="rendered patch content & style feature loss")
    parser.add_argument("--PCSLoss_C_lamb", type=float, default=1e-3, help="rendered patch content feature loss lambda")
    parser.add_argument("--PCSLoss_S_lamb", type=float, default=1e-4, help="rendered patch style feature loss lambda")
    parser.add_argument("--patch_nerf_depth_tvLoss", action="store_true", help="patch nerf depth tv loss") # refer to RC-MVS
    parser.add_argument("--patch_nerf_depth_tvLoss_lamb", type=float, default=0.1, help="patch nerf depth tv loss lambda")
    parser.add_argument("--inputWhole_cs_loss", action="store_true", help="stylized input(interpolate feature by pred_depth -> MLP) content & style feature loss")
    parser.add_argument("--ICSLoss_C_lamb", type=float, default=5e-3, help="stylized input content feature loss lambda")
    parser.add_argument("--ICSLoss_S_lamb", type=float, default=1e-4, help="stylized input style feature loss lambda")
    parser.add_argument("--wo_style_loss", action="store_true", help="without style novel view loss")
    parser.add_argument("--input_style_loss", action="store_true", help="style input view0 loss")
    parser.add_argument("--timephi_loss", action="store_true", help="time phi loss")
    parser.add_argument("--phi_loss_lamb", type=float, default=100, help="time phi loss")
    parser.add_argument("--STB_style_loss", action="store_true", help="style two branch loss(common+sepcial->enc_a <-> style->enc_a)")
    parser.add_argument("--STB_loss_lamb", type=float, default=5e-3, help="style two branch loss(common+sepcial->enc_a <-> style->enc_a)")
    parser.add_argument("--STB_style_mse_loss", action="store_true", help="style two branch loss(stylized input mse loss)")
    parser.add_argument("--style_D", action="store_true", help="style two branch -> discriminator")
    parser.add_argument("--style_D_lamb", type=float, default=1e-2, help="style_D gan loss lambda")
    parser.add_argument("--weatherPred_lamb", type=float, default=1e-2, help="weather pred lambda")
    parser.add_argument("--style_D_mdmm", action="store_true", help="style two branch -> MDMM discriminator")
    parser.add_argument("--style_D_mdmm_lamb", type=float, default=1e-2, help="style_D gan loss lambda")
    parser.add_argument("--branch2_cycle_loss", action="store_true", help="branch2 cycle loss")
    parser.add_argument("--branch2_cycle_lamb", type=float, default=0.1, help="branch2 cycle loss lambda")
    parser.add_argument("--input_rec_loss", action="store_true", help="input img(a) -> phi -> stylize input img(b) => (a)=(b)")
    parser.add_argument("--input_rec_lamb", type=float, default=0.5, help="input rec loss lambda")
    parser.add_argument("--t0_rec_loss", action="store_true", help="input image(t0) go through MLPstyle")
    parser.add_argument("--t0_rec_lamb", type=float, default=0.5, help="t0 rec loss lambda")
    parser.add_argument("--delta_t_loss_lamb", type=float, default=0.01, help="delta_t_loss lambda")
    parser.add_argument("--t_regularization", action="store_true", help="t_regularization")
    parser.add_argument("--t_regularization_lamb", type=float, default=0.005, help="t_regularization loss lambda")
    parser.add_argument("--rgb2t", action="store_true", help="rgb -> cos(t),sin(t)")
    parser.add_argument("--rgb2t_loss_lamb", type=float, default=0.005, help="rgb2t loss lambda")

    # Rendering options
    parser.add_argument("--chunk", type=int, default=4096, help="Number of rays rendered in parallel")
    parser.add_argument("--nb_coarse", type=int, default=96, help="Number of coarse samples per ray")
    parser.add_argument("--nb_fine", type=int, default=32, help="Number of additional fine samples per ray",)
    parser.add_argument("--tune_result_by_delta_t", action="store_true", help="tune result by delta t")

    # Other options
    parser.add_argument("--seed", type=int, default=20, help="seed")
    parser.add_argument("--expname", type=str, help="Experiment name")
    parser.add_argument("--logger", type=str, default="none", choices=["wandb", "tensorboard", "none"])
    parser.add_argument("--logdir", type=str, default="./logs/", help="Where to store ckpts and logs")
    parser.add_argument("--eval", action="store_true", help="Render and evaluate the test set")
    parser.add_argument("--finetune", action="store_true", help="use pretrain ver0")
    parser.add_argument("--freeze", action="store_true", help="freeze pretrain ver0")
    parser.add_argument("--use_depth", action="store_true", help="Use ground truth low-res depth maps in rendering process")
    parser.add_argument("--save_sigma", action="store_true", help="save sigma")
    parser.add_argument("--save_T", action="store_true", help="save T")
    parser.add_argument("--save_nerf_rgb", action="store_true", help="save nerf_rgb")
    parser.add_argument("--save_alpha", action="store_true", help="save alpha")
    parser.add_argument("--save_coord", action="store_true", help="save coord")
    parser.add_argument("--save_D", action="store_true", help="save disocclusion confidence")
    parser.add_argument("--save_O_label", action="store_true", help="save O label")
    parser.add_argument("--save_O", action="store_true", help="save O (geometry feature)")
    parser.add_argument("--save_depth_prob", action="store_true", help="save depth prob")
    parser.add_argument("--ycbcr", action="store_true", help="img: YCbCr")
    parser.add_argument("--save_input_depth", action="store_true", help="save input depth pred (casMVS)")
    parser.add_argument("--save_input_img", action="store_true", help="save input imgs")
    parser.add_argument("--edge", action="store_true", help="detect edge")
    parser.add_argument("--sample", action="store_true", help="sample distribution")
    parser.add_argument("--sample_num", type=int, default=3, help="sample distribution times")
    parser.add_argument("--gene_mask", type=str, default="None", choices=["interval", "one_pt", "None"], help="generate mask (interval: check points in an interval; one_pt: check one point(depth GT))")
    parser.add_argument("--visMask", type=str, default="None", help="if not None: use genemask to evaluate", choices=["vis", "invis", "None"])
    parser.add_argument("--visMask_type", type=str, default="None", help="if not None: use genemask to evaluate", choices=["interval", "one_pt", "None"])
    parser.add_argument("--occ_mask_useGT", action="store_true", help="occ_mask use depth gt")
    parser.add_argument("--oneImg", action="store_true", help="save one image")
    parser.add_argument("--oneImg_what", type=str, default="nvs", help="save one image for ??")
    parser.add_argument("--eval_from_img", action="store_true", help="eval from images")
    parser.add_argument("--eval_from_img_path", type=str, default="./lama/output", help="images path to eval")
    parser.add_argument("--dont_save_eval", action="store_true", help="don't save eval")
    parser.add_argument("--save_mask", action="store_true", help="save mask (depth>0 & vis_mask)")
    parser.add_argument("--save_var_confi", type=int, default=-1, help="save var confi")
    parser.add_argument("--save_pred_cls_loss", action="store_true", help="save MDMM pred cls loss")
    parser.add_argument("--save_pred_phi", action="store_true", help="save predicted time phi")
    parser.add_argument("--input_phi_to_test", action="store_true", help="input time phi to test")
    parser.add_argument("--phi_is_from_input", action="store_true", help="input time phi is extracted from input")
    parser.add_argument("--psnr_t0", action="store_true", help="evaluate nvs metric on t0 images")

    
    geonerf_args, unknown = parser.parse_known_args()
    if geonerf_args.geonerfMDMM or geonerf_args.contentFeature:
        from MDMM.options import TestOptions
        from argparse import Namespace
        MDMM_parser = TestOptions(parser=parser)
        MDMM_args = MDMM_parser.parse()

        args_dict, MDMM_args_dict = vars(geonerf_args), vars(MDMM_args)
        args_dict.update(MDMM_args_dict)
        args = Namespace(**args_dict)

        return args
    else:
        return parser.parse_args()
