#https://github.com/lucidrains/vit-pytorch/blob/main/vit_pytorch/simple_vit.py
from copy import deepcopy
import torch
from torch import nn
from einops import rearrange
from einops.layers.torch import Rearrange

from utility.args import Args

Args.add_argument("--ViTPatchSize", type=int, help="The image is tiled into patches of size ViTPatchSize x ViTPatchSize. Each patch gets an embedding.")
Args.add_argument("--ViTSize", type=str, help="Vision Transformer size named after original paper; we support Ti(ny) S(mall) and B(ig).")


# helpers

def pair(t):
    return t if isinstance(t, tuple) else (t, t)

def posemb_sincos_2d(patches, temperature = 10000, dtype = torch.float32):
    _, h, w, dim, device, dtype = *patches.shape, patches.device, patches.dtype

    y, x = torch.meshgrid(torch.arange(h, device = device), torch.arange(w, device = device), indexing = 'ij')
    assert (dim % 4) == 0, 'feature dimension must be multiple of 4 for sincos emb'
    omega = torch.arange(dim // 4, device = device) / (dim // 4 - 1)
    omega = 1. / (temperature ** omega)

    y = y.flatten()[:, None] * omega[None, :]
    x = x.flatten()[:, None] * omega[None, :] 
    pe = torch.cat((x.sin(), x.cos(), y.sin(), y.cos()), dim = 1)
    return pe.type(dtype)

# classes

class FeedForward(nn.Module):
    def __init__(self, dim, hidden_dim):
        super().__init__()
        self.net = nn.Sequential(
            nn.LayerNorm(dim),
            nn.Linear(dim, hidden_dim),
            nn.GELU(),
            nn.Linear(hidden_dim, dim),
        )
    def forward(self, x):
        return self.net(x)

class Attention(nn.Module):
    def __init__(self, dim, heads = 8, dim_head = 64):
        super().__init__()
        inner_dim = dim_head *  heads
        self.heads = heads
        self.scale = dim_head ** -0.5
        self.norm = nn.LayerNorm(dim)

        self.attend = nn.Softmax(dim = -1)

        self.to_qkv = nn.Linear(dim, inner_dim * 3, bias = False)
        self.to_out = nn.Linear(inner_dim, dim, bias = False)

    def forward(self, x):
        x = self.norm(x)

        qkv = self.to_qkv(x).chunk(3, dim = -1)
        q, k, v = map(lambda t: rearrange(t, 'b n (h d) -> b h n d', h = self.heads), qkv)

        dots = torch.matmul(q, k.transpose(-1, -2)) * self.scale

        attn = self.attend(dots)

        out = torch.matmul(attn, v)
        out = rearrange(out, 'b h n d -> b n (h d)')
        return self.to_out(out)

class Transformer(nn.Module):
    def __init__(self, dim, depth, heads, dim_head, mlp_dim): # , sideOutputDepth = 1):
        super().__init__()
        #self.sideOutputIndex = depth - sideOutputDepth
        self.layers = nn.ModuleList([])
        for _ in range(depth):
            self.layers.append(nn.ModuleList([
                Attention(dim, heads = heads, dim_head = dim_head),
                FeedForward(dim, mlp_dim)
            ]))
        
    def forward(self, x):
        #sideOut = None
        for i, (attn, ff) in enumerate(self.layers):
            x = attn(x) + x
            x = ff(x) + x
            #if self.sideOutputIndex == i:
                #sideOut = x
        return x #, sideOut

class SimpleViT(nn.Module):
    def __init__(self, num_classes):
        super().__init__()
        if Args.imageSize == 0:
            raise ValueError("Argument imageSize must be different from 0.")
        image_size = Args.imageSize
        patch_size = Args.ViTPatchSize
        self.num_classes = num_classes
        self.subLabels = Args.subLabels

        #https://arxiv.org/pdf/2106.04560.pdf table 2
        architecture_parameters = {
            "Ti": (192, 12, 768, 3),
            "S": (384, 12, 1536, 6),
            "B": (768, 12, 3072, 12),
        }
        if not Args.ViTSize in architecture_parameters:
            raise RuntimeError(f"Argument ViTSize must be in {', '.join(architecture_parameters.keys())} but is {Args.ViTSize}.")
        dim, depth, mlp_dim, heads = architecture_parameters[Args.ViTSize]

        channels = 3
        dim_head = 64

        image_height, image_width = pair(image_size)
        patch_height, patch_width = pair(patch_size)

        if not (image_height % patch_height == 0 and image_width % patch_width == 0):
            raise ValueError("(Both) Image dimensions must be divisible by the patch size.")

        patch_dim = channels * patch_height * patch_width

        self.to_patch_embedding = nn.Sequential(
            Rearrange('b c (h p1) (w p2) -> b h w (p1 p2 c)', p1 = patch_height, p2 = patch_width),
            nn.LayerNorm(patch_dim),
            nn.Linear(patch_dim, dim),
            nn.LayerNorm(dim),
        )

        self.transformer = Transformer(dim, depth, heads, dim_head, mlp_dim)  #, sideOutputDepth = Args.copyDepth)

        self.finalLayerNorm = nn.LayerNorm(dim)
        self.classification = nn.Linear(dim, num_classes)

        dimensions = [dim] + Args.rndHeadHL + [Args.subLabels]
        layers = []
        for i in range(len(dimensions) - 1):
            layers.append(nn.Linear(dimensions[i], dimensions[i+1]))
            if i != len(dimensions) - 2:
                layers.append(nn.ReLU(inplace=True))
        singleHead = nn.Sequential(*layers)
        if Args.singleHead:
            self.subPredictionsNet = singleHead
        else:
            self.subPredictionsNet = nn.ModuleList([deepcopy(singleHead) for _ in range(self.num_classes)])

        self.penultimate = None

    def forward(self, img):
        *_, h, w, dtype = *img.shape, img.dtype

        x = self.to_patch_embedding(img)
        pe = posemb_sincos_2d(x)
        x = rearrange(x, 'b ... d -> b (...) d') + pe

        x = self.transformer(x)  #x, sideOut = self.transformer(x)
        x = x.mean(dim = 1)
        #sideOut = sideOut.mean(dim = 1)

        self.penultimate = self.finalLayerNorm(x)

        if Args.singleHead:
            out_ = self.subPredictionsNet(self.penultimate)
        else:
            branches = []
            for i in range(self.num_classes):
                branches.append(self.subPredictionsNet[i](self.penultimate))
            out_ = torch.concat(branches, dim = 1)

        return out_, self.classification(self.penultimate)
    
    
    def setBatchNormTracking(self, track_running_stats: bool):
        for m in self.modules():
            if isinstance(m, nn.modules.batchnorm._BatchNorm):
                m.track_running_stats = track_running_stats