""" https://github.com/xternalz/WideResNet-pytorch/blob/master/wideresnet.py """
from copy import deepcopy
import torch
import torch.nn as nn
import torch.nn.functional as F

from utility.args import Args

Args.add_argument("--widthFactor", type=float, help="How many times wider compared to normal ResNet.")

class BasicBlock(nn.Module):
    def __init__(self, in_planes, out_planes, stride, dropRate=0.0):
        super(BasicBlock, self).__init__()
        self.bn1 = nn.BatchNorm2d(in_planes)
        self.relu1 = nn.ReLU(inplace=True)
        self.conv1 = nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride,
                               padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(out_planes)
        self.relu2 = nn.ReLU(inplace=True)
        self.conv2 = nn.Conv2d(out_planes, out_planes, kernel_size=3, stride=1,
                               padding=1, bias=False)
        self.droprate = dropRate
        self.equalInOut = (in_planes == out_planes) and stride == 1
        self.shortcut = (not self.equalInOut) and nn.Conv2d(in_planes, out_planes, kernel_size=1, stride=stride,
                               padding=0, bias=False) or torch.nn.Identity()
    def forward(self, x):
        if not self.equalInOut:
            x = self.bn1(x)
            x = self.relu1(x)
        else:
            out = self.relu1(self.bn1(x))
        out = self.relu2(self.bn2(self.conv1(out if self.equalInOut else x)))
        if self.droprate > 0:
            out = F.dropout(out, p=self.droprate, training=self.training)
        out = self.conv2(out)
        return torch.add(self.shortcut(x), out)

class NetworkBlock(nn.Module):
    def __init__(self, nb_layers, in_planes, out_planes, block, stride, dropRate=0.0):
        super(NetworkBlock, self).__init__()
        self.layer = self._make_layer(block, in_planes, out_planes, nb_layers, stride, dropRate)
    def _make_layer(self, block, in_planes, out_planes, nb_layers, stride, dropRate):
        layers = []
        for i in range(int(nb_layers)):
            layers.append(block(i == 0 and in_planes or out_planes, out_planes, i == 0 and stride or 1, dropRate))
        return nn.Sequential(*layers)
    def forward(self, x):
        return self.layer(x)

class WideResNet(nn.Module):
    def __init__(self, num_classes):
        super(WideResNet, self).__init__()

        dropRate = Args.dropout
        widen_factor = Args.widthFactor
        depth = Args.depth

        self.num_classes = num_classes
        self.subLabels = Args.subLabels

        self.allowFactors = [1/8, 1/16, 1/32, 1/64] # Allowed values smaller .25 for Args.widthFactor (=widen_factor).

        if (depth - 4) % 6 != 0:
            raise RuntimeError(f"Depth = {depth} does not follow 6*n + 4, n \\in N")

        if widen_factor in self.allowFactors:
            UserWarning(f"Be aware that with an 'Args.widthFactor' smaller than .25, the first conv layer and possibly "
                        f"other layers' output channels cannot be scaled. "
                        f"In this case, the number of output channels for, e.g., conv1 is always equal to 1. "
                        f"Your width factor: {widen_factor}. ")
        #elif widen_factor % .25 != 0:
        #    raise RuntimeError(f"WRN width factor = {widen_factor} (Args.widthFactor) has to be divisible by 0.25 or "
        #                       f"it must hold Args.widthFactor \\in {self.allowFactors}.")

        # Original WRN paper implementation (be aware that He et al. always use 64 channels for the first conv layer).
        # This is different to the double descent paper (as well as mentioned to the original ResNet paper).
        # nChannels = [16, 16*widen_factor, 32*widen_factor, 64*widen_factor]
        # Implementation for double descent experiments (default behavior for widen_factor=4 is unchanged!).
        nChannels = [max(1, int(4 * widen_factor)), max(1, int(16 * widen_factor)), max(1, int(32 * widen_factor)), int(64 * widen_factor)]

        n = (depth - 4) / 6
        block = BasicBlock
        # 1st conv before any network block
        self.conv1 = nn.Conv2d(3, nChannels[0], kernel_size=3, stride=1,
                               padding=1, bias=False)
        # 1st block
        self.block1 = NetworkBlock(n, nChannels[0], nChannels[1], block, 1, dropRate)
        # 2nd block
        self.block2 = NetworkBlock(n, nChannels[1], nChannels[2], block, 2, dropRate)
        # 3rd block
        self.block3 = NetworkBlock(n, nChannels[2], nChannels[3], block, 2, dropRate)
        # global average pooling and classifier
        self.nChannels = nChannels[3]
        self.bn1 = nn.BatchNorm2d(self.nChannels)
        self.relu = nn.ReLU(inplace=True)
        self.classification = nn.Linear(self.nChannels, num_classes)

        dimensions = [self.nChannels] + Args.rndHeadHL + [Args.subLabels]
        layers = []
        for i in range(len(dimensions) - 1):
            layers.append(nn.Linear(dimensions[i], dimensions[i+1]))
            if i != len(dimensions) - 2:
                layers.append(nn.ReLU(inplace=True))
        singleHead = nn.Sequential(*layers)
        if Args.singleHead:
            self.subPredictionsNet = singleHead
        else:
            self.subPredictionsNet = nn.ModuleList([deepcopy(singleHead) for _ in range(self.num_classes)])

        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
            elif isinstance(m, nn.BatchNorm2d):
                m.weight.data.fill_(1)
                m.bias.data.zero_()
            elif isinstance(m, nn.Linear):
                m.bias.data.zero_()

        self.penultimate = None

    def setBatchNormTracking(self, track_running_stats: bool):
        for m in self.modules():
            if isinstance(m, nn.modules.batchnorm._BatchNorm):
                m.track_running_stats = track_running_stats

    def forward(self, x):
        out = self.conv1(x)
        out = self.block1(out)
        out = self.block2(out)
        out = self.block3(out)
        out = self.bn1(out)
        out = self.relu(out)
        out = F.avg_pool2d(out, 8, 1, 0)
        out = out.view(-1, self.nChannels)
        
        self.penultimate = out

        if Args.singleHead:
            out_ = self.subPredictionsNet(out)
        else:
            branches = []
            for i in range(self.num_classes):
                branches.append(self.subPredictionsNet[i](out))
            out_ = torch.concat(branches, dim = 1)
        return out_, self.classification(out)
