import time
import functools
from contextlib import contextmanager

class ProfilerTimer:
    """Simple profiler for debugging performance bottlenecks

    Usage:
        # Initialize in main function
        profiler = ProfilerTimer(enabled=bool(DISTRL_DEBUG_PROFILING), is_main_process=accelerator.is_local_main_process)

        # Use start/end methods
        profiler.start("operation_name")
        # your code here
        profiler.end("operation_name")

        # Use as function decorator
        @profiler.profile_function("custom_name")
        def my_function():
            pass
    """

    def __init__(self, enabled=False, is_main_process=True):
        self.enabled = enabled and is_main_process
        self.timers = {}

    def start(self, name):
        """Start timing for the given operation name"""
        if not self.enabled:
            return
        self.timers[name] = time.perf_counter()

    def end(self, name):
        """End timing for the given operation name and print duration"""
        if not self.enabled:
            return
        if name in self.timers:
            duration = time.perf_counter() - self.timers[name]
            print(f"[PROFILING] {name}: {duration:.4f}s")
            del self.timers[name]

    @contextmanager
    def timer(self, name):
        """Context manager for timing code blocks (kept for compatibility)"""
        if not self.enabled:
            yield
            return

        start_time = time.perf_counter()
        try:
            yield
        finally:
            end_time = time.perf_counter()
            duration = end_time - start_time
            print(f"[PROFILING] {name}: {duration:.4f}s")

    def profile_function(self, name=None):
        """Decorator for timing functions"""
        def decorator(func):
            if not self.enabled:
                return func

            func_name = name or f"{func.__module__}.{func.__name__}"

            @functools.wraps(func)
            def wrapper(*args, **kwargs):
                start_time = time.perf_counter()
                try:
                    result = func(*args, **kwargs)
                    return result
                finally:
                    end_time = time.perf_counter()
                    duration = end_time - start_time
                    print(f"[PROFILING] {func_name}: {duration:.4f}s")
            return wrapper
        return decorator

class FakeProfilerTimer:
    def __init__(self):
        pass

    def start(self, name):
        pass

    def end(self, name):
        pass
