import regex
from abc import ABC, abstractmethod
import json
import re
import logging
import logging_config

class BaseParser(ABC):
    def __init__(self):
        self.error = ""
        self.name = ""

    @abstractmethod
    def parse(self, input: str):
        '''
        Parse the input. If fail, return `None`.
        '''
        pass

    def to_dict(self) -> dict:
        return {
            "name": self.name
        }

    def print_error(self) -> str:
        return f"Parser-{self.name} Error: {self.error}"
    
class DefaultParser(BaseParser):
    def __init__(self):
        super().__init__()
        self.name = "Default"

    def parse(self, input: str):
        return input

class JsonParser(BaseParser):
    def __init__(self):
        super().__init__()
        self.name = "Json"

    def parse(self, input: str):
        self.error = ""
        if input == None:
            self.error = "[Invalid raw LLM response]"
            return None
        try:
            json_pattern = r'\{(?:[^{}]|(?R))*\}'
            json_match = regex.search(json_pattern, input)
            if json_match == None:
                raise TypeError
            json_str = json_match.group()
            extracted_json = json.loads(json_str)
            return extracted_json
        except Exception:
            self.error = "[No JSON found]"
            return None

class PythonParser(BaseParser):
    def __init__(self):
        super().__init__()
        self.name = "Python"
    
    def parse(self, input: str):
        self.error = ""
        if input == None:
            self.error = "[Invalid raw LLM response]"
            return None
        match = re.search(r'```python(.*?)```', input, re.DOTALL)
        if match:
            code = match.group(1).strip()
            if len(code) > 0:
                return code
        self.error = "[No Python code found]"
        return None