from KeywordList import KeywordList
from BaseDesign import BaseDesign
from enum import Enum
import json
from typing import List

class EntryType(Enum):
    RSDICT=0
    RSDICT_SF=1
    PLANSEARCH=2
    RS=3
    RSDICT_NOWR=4
    RSDICT_SF_NOWR=5

class Entry:
    def __init__(self, id: int, entry_type: EntryType):
        '''
        Initialize an entry with:
        - id
        - entry_type
        '''
        self.id: int = id
        self.entry_type: EntryType = entry_type
        self.hints: KeywordList = None
        self.design: BaseDesign = None
        self.code: str = None # path to the code
        self.feedback_embedding: List[float] = None

    def __str__(self):
        return f"(id={self.id}, type={self.entry_type.name})"
    def __repr__(self):
        return f"(id={self.id}, type={self.entry_type.name})"

    def set_hints(self, hints_str: str):
        self.hints = KeywordList(hints_str) if hints_str != None else None

    def set_design(self, design_dict: dict):
        self.design = type(self.design).from_dict(design_dict) if design_dict != None else None

    def set_code(self, code: str):
        self.code = code

    def set_feedback_embedding(self, feedback_embedding: list):
        self.feedback_embedding = feedback_embedding
        
    @classmethod
    def from_dict(cls, entry_dict):
        entry = cls(
            id=entry_dict["id"],
            entry_type=getattr(type(cls.entry_type), entry_dict["type"]),
        )
        entry.set_hints(entry_dict["hints"])
        entry.set_design(entry_dict["design"])
        entry.set_code(entry_dict["code"])
        entry.set_feedback_embedding(entry_dict["feedback_embedding"])
        return entry
    
    @classmethod
    def from_jsonl(cls, jsonl: str):
        return cls.from_dict(entry_dict=json.loads(jsonl))

    def to_dict(self) -> dict:
        entry_dict = {
            "id": self.id,
            "type": self.entry_type.name,
            "hints": self.hints.to_str() if self.hints != None else None,
            "design": self.design.to_dict() if self.design != None else None,
            "code": self.code,
            "feedback_embedding": self.feedback_embedding,
        }
        return entry_dict

    def to_jsonl(self) -> str:
        return json.dumps(self.to_dict())
