from Simulator import SimulatorCache
from BaseParser import BaseParser, JsonParser, PythonParser
    
class MutateParser(BaseParser):
    def __init__(self):
        super().__init__()
        self.name = "Mutate"

    def parse(self, input: str):
        if input == None:
            self.error = "[Invalid raw LLM response]"
            return None
        parts = input.split(":::")
        if len(parts) == 1:
            return input
        return parts[-1].strip()
    
class MapParser(BaseParser):
    def __init__(self):
        super().__init__()
        self.name = "Map"

    def parse(self, input: str):
        if input == None:
            self.error = "[Invalid raw LLM response]"
            return None
        lines = input.strip().split("\n")
        nonempty_lines = [l.strip() for l in lines if len(l.strip()) > 0]
        if len(nonempty_lines) == 0:
            self.error = "[No Answer found]"
            return None
        target_line = nonempty_lines[-1]
        return target_line
            
class DesignJsonParser(JsonParser):
    def __init__(self):
        super().__init__()
        self.name = "DesignJson"

    def parse(self, input: str):
        extracted_json = super().parse(input)
        if extracted_json != None:
            design_json_error = ""
            if "metadata" not in extracted_json:
                design_json_error += " metadata"
            if "evict" not in extracted_json:
                design_json_error += " evict"
            if "update_after_hit" not in extracted_json:
                design_json_error += " update_after_hit"
            if "update_after_insert" not in extracted_json:
                design_json_error += " update_after_insert"
            if "update_after_evict" not in extracted_json:
                design_json_error += " update_after_evict"
            if design_json_error != "":
                self.error = "[Invalid JSON format]" + design_json_error
                return None
            return extracted_json
        return None
    
class CacheCodeParser(PythonParser):
    def __init__(self, unique_simulator: SimulatorCache):
        super().__init__()
        self.name = "CacheCode"
        self.simulator = unique_simulator
        self.code_id = None

    def set_code_id(self, code_id):
        '''
        Must be executed before parsing a LLM-generated code.
        '''
        self.code_id = code_id
    
    def parse(self, input: str):
        assert self.code_id != None
        code = super().parse(input)
        if code != None:
            default_miss_ratio = self.simulator.simulate(code, self.code_id)
            if default_miss_ratio == None:
                self.error = "[Invalid Cache Code]"
                return None
            else:
                # tune, failure is acceptable
                tuned_miss_ratio, default_params, tuned_params = self.simulator.tune(code, self.code_id, False)
                if tuned_miss_ratio == None or tuned_miss_ratio > default_miss_ratio:
                    tuned_miss_ratio = default_miss_ratio
                    tuned_params = default_params
                return default_miss_ratio, tuned_miss_ratio, default_params, tuned_params
        else:
            return None
        
    def to_dict(self) -> dict:
        my_dict = super().to_dict()
        my_dict.update({"simulator": self.simulator.to_dict()})
        return my_dict
