#!/usr/bin/env python
# coding=utf-8
# Copyright 2020 The HuggingFace Inc. team. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""
Fine-tuning the library models for causal language modeling (GPT, GPT-2, CTRL, ...) on a text file or a dataset.
Here is the full list of checkpoints on the hub that can be fine-tuned by this script:
https://huggingface.co/models?filter=text-generation
"""
# You can also adapt this script on your own causal language modeling task. Pointers for this are left as comments.

import logging
import math
import os
os.environ["WANDB_DISABLED"] = "true"


import sys
from dataclasses import dataclass, field
from itertools import chain
from typing import Optional, List 
import re
import datasets
import evaluate
import torch
from datasets import load_dataset, concatenate_datasets
import numpy as np
import random
import transformers
from transformers import (
    CONFIG_MAPPING,
    MODEL_FOR_CAUSAL_LM_MAPPING,
    AutoConfig,
    AutoModelForCausalLM,
    AutoTokenizer,
    LlamaTokenizer,
    LlamaTokenizerFast,
    HfArgumentParser,
    Trainer,
    TrainingArguments,
    default_data_collator,
    is_torch_tpu_available,
    set_seed,
    get_scheduler,
    DataCollatorForSeq2Seq,
    GPTNeoXTokenizerFast,
    GPT2Tokenizer,
    OPTForCausalLM,
    TrainerCallback
)
from transformers.testing_utils import CaptureLogger
from transformers.trainer_utils import get_last_checkpoint
from transformers.utils import check_min_version, send_example_telemetry
from transformers.utils.versions import require_version

from torch.utils.data import DataLoader
from accelerate import Accelerator
from accelerate.checkpointing import save_accelerator_state
from accelerate.logging import get_logger
from accelerate.utils import set_seed

from peft.tuners.lora import LoraLayer
from lib.get_dataset import get_dataset_c4, get_dataset_sft, get_dataset_alpaca, WarmupStableDrop, CombinedDataset, DoubleWSD

from peft import (
    LoraConfig,
    get_peft_model,
    get_peft_model_state_dict,
    prepare_model_for_int8_training,
    set_peft_model_state_dict,
    SftConfig,
    SftTrainer,
    SftAdamW,
    SftSelector,
    VeraConfig
)


# Will error if the minimal version of Transformers is not installed. Remove at your own risks.
# check_min_version("4.29.0.dev0")

# require_version("datasets>=1.8.0", "To fix: pip install -r examples/pytorch/language-modeling/requirements.txt")

logger = logging.getLogger(__name__)


MODEL_CONFIG_CLASSES = list(MODEL_FOR_CAUSAL_LM_MAPPING.keys())
MODEL_TYPES = tuple(conf.model_type for conf in MODEL_CONFIG_CLASSES)

def str2bool(v):
    """
    Converts string to bool type; enables command line
    arguments in the format of '--arg1 true --arg2 false'
    """
    if isinstance(v, bool):
        return v
    if v.lower() in ('yes', 'true', 't', 'y', '1'):
        return True
    elif v.lower() in ('no', 'false', 'f', 'n', '0'):
        return False
    else:
        raise argparse.ArgumentTypeError('Boolean value expected.')


@dataclass
class ModelArguments:
    """
    Arguments pertaining to which model/config/tokenizer we are going to fine-tune, or train from scratch.
    """

    model_name_or_path: Optional[str] = field(
        default=None,
        metadata={
            "help": (
                "The model checkpoint for weights initialization.Don't set if you want to train a model from scratch."
            )
        },
    )
    model_type: Optional[str] = field(
        default=None,
        metadata={"help": "If training from scratch, pass a model type from the list: " + ", ".join(MODEL_TYPES)},
    )
    peft_type: Optional[str] = field(
        default=None,
        metadata={"help": "lora or sft"},
    )

    lora_r: Optional[int] = field(
        default=8,
        metadata={"help": "parameter lora_r"},
    )
    lora_alpha: Optional[int] = field(
        default=16,
        metadata={"help": "parameter lora_alpha"},
    )
    lora_dropout: Optional[float] = field(
        default=0.05,
        metadata={"help": "parameter lora_dropout"},
    )

    sft_density: Optional[float] = field(
        default=None,
        metadata={"help": "Density of sparse fine-tuning"},
    )
    sft_num_deltas: Optional[int] = field(
        default=None,
        metadata={"help": "Total number of tunable parameters in sparse fine-tuning (overrides --sft_density)."},
    )
    sft_reselection_steps: Optional[int] = field(
        default=20,
        metadata={"help": "Number of grad accumulation steps per SFT candidate reselection."},
    )
    sft_selection_accumulation_steps: Optional[int] = field(
        default=5,
        metadata={"help": "Number of grad accumulation steps per SFT candidate reselection."},
    )
    sft_selection: Optional[str] = field(
        default="rigl",
        metadata={"help": "Algorithm for selection SFT delta params: 'rigl' or 'sm3'."},
    )
    sparse_alg: Optional[str] = field(
        default="dst",
        metadata={"help": "Algorithm for selection SFT delta params: 'dst' or 'fix'."},
    )
    # lora_target_modules: Optional[List[str]] = field(
    #     default=["q_proj","v_proj"],
    #     metadata={"help": "parameter lora_target_modules"},
    # )
    config_overrides: Optional[str] = field(
        default=None,
        metadata={
            "help": (
                "Override some existing default config settings when a model is trained from scratch. Example: "
                "n_embd=10,resid_pdrop=0.2,scale_attn_weights=false,summary_type=cls_index"
            )
        },
    )
    config_name: Optional[str] = field(
        default=None, metadata={"help": "Pretrained config name or path if not the same as model_name"}
    )
    tokenizer_name: Optional[str] = field(
        default=None, metadata={"help": "Pretrained tokenizer name or path if not the same as model_name"}
    )
    cache_dir: Optional[str] = field(
        default="",
        metadata={"help": "Where do you want to store the pretrained models downloaded from huggingface.co"},
    )
    use_fast_tokenizer: bool = field(
        default=True,
        metadata={"help": "Whether to use one of the fast tokenizer (backed by the tokenizers library) or not."},
    )
    model_revision: str = field(
        default="main",
        metadata={"help": "The specific model version to use (can be a branch name, tag name or commit id)."},
    )
    use_auth_token: bool = field(
        default=False,
        metadata={
            "help": (
                "Will use the token generated when running `huggingface-cli login` (necessary to use this script "
                "with private models)."
            )
        },
    )
    torch_dtype: Optional[str] = field(
        default=None,
        metadata={
            "help": (
                "Override the default `torch.dtype` and load the model under this dtype. If `auto` is passed, the "
                "dtype will be automatically derived from the model's weights."
            ),
            "choices": ["auto", "bfloat16", "float16", "float32"],
        },
    )
    low_cpu_mem_usage: bool = field(
        default=False,
        metadata={
            "help": (
                "It is an option to create the model as an empty shell, then only materialize its parameters when the pretrained weights are loaded."
                "set True will benefit LLM loading time and RAM consumption."
            )
        },
    )

    mask_zeros: str2bool = field(
        default=False,
        metadata={
            "help": (
                "Will use the token generated when running `huggingface-cli login` (necessary to use this script "
                "with private models)."
            )
        },
    )

    pruned: str2bool = field(
        default=False,
        metadata={
            "help": (
                "Will use the token generated when running `huggingface-cli login` (necessary to use this script "
                "with private models)."
            )
        },
    )

    use_qpeft: str2bool = field(
        default=False,
        metadata={
            "help": (
                "Will use the token generated when running `huggingface-cli login` (necessary to use this script "
                "with private models)."
            )
        },
    )

    ## owl
    merge_ratio: Optional[float] = field(
        default=0.1,
        metadata={"help": "parameter lora_dropout"},
    )
    sparsity_ratio: Optional[float] = field(
        default=0.5,
        metadata={"help": "parameter lora_dropout"},
    )
    Lamda: Optional[float] = field(
        default=0.08,
        metadata={"help": "parameter lora_dropout"},
    )
    using_allocate: str = field(
        default='uniform',
        metadata={"help": "parameter lora_dropout"},
    )

    target_modules_model: Optional[str] = field(
        default=None,  ## None
        metadata={
            "help": "List of module names or regex expression of the module names to replace with Lora."
                    "For example, ['q', 'v'] or '.*decoder.*(SelfAttention|EncDecAttention).*(q|v)$' "
        },
    )

    initial_reselection_rate: Optional[float] = field(
        default=0.2,
        metadata={"help": "initial_reselection_rate"},
    )


    ## memory
    max_memory: Optional[int] = field(
        default=-1,
        metadata={"help": "Allows to limit the memory on the cuda device that can be used."},
    )
    '''
    gradient_checkpointing: str2bool = field(
        default=False,
        metadata={
            "help": "Turn on gradient checkpointing. Saves memory but slows training."
        },
    )
    '''
    def __post_init__(self):
        if self.config_overrides is not None and (self.config_name is not None or self.model_name_or_path is not None):
            raise ValueError(
                "--config_overrides can't be used in combination with --config_name or --model_name_or_path"
            )


@dataclass
class DataTrainingArguments:
    """
    Arguments pertaining to what data we are going to input our model for training and eval.
    """

    dataset_name: Optional[str] = field(
        default='c4', metadata={"help": "The name of the dataset to use (via the datasets library)."}
    )
    dataset_config_name: Optional[str] = field(
        default=None, metadata={"help": "The configuration name of the dataset to use (via the datasets library)."}
    )
    train_file: Optional[str] = field(default=None, metadata={"help": "The input training data file (a text file)."})
    validation_file: Optional[str] = field(
        default=None,
        metadata={"help": "An optional input evaluation data file to evaluate the perplexity on (a text file)."},
    )
    max_train_samples: Optional[int] = field(
        default=None,
        metadata={
            "help": (
                "For debugging purposes or quicker training, truncate the number of training examples to this "
                "value if set."
            )
        },
    )
    max_eval_samples: Optional[int] = field(
        default=None,
        metadata={
            "help": (
                "For debugging purposes or quicker training, truncate the number of evaluation examples to this "
                "value if set."
            )
        },
    )
    streaming: bool = field(default=False, metadata={"help": "Enable streaming mode"})
    block_size: Optional[int] = field(
        default=None,
        metadata={
            "help": (
                "Optional input sequence length after tokenization. "
                "The training dataset will be truncated in block of this size for training. "
                "Default to the model max input length for single sentence inputs (take into account special tokens)."
            )
        },
    )
    overwrite_cache: bool = field(
        default=False, metadata={"help": "Overwrite the cached training and evaluation sets"}
    )
    validation_split_percentage: Optional[int] = field(
        default=5,
        metadata={
            "help": "The percentage of the train set used as validation set in case there's no validation split"
        },
    )
    preprocessing_num_workers: Optional[int] = field(
        default=None,
        metadata={"help": "The number of processes to use for the preprocessing."},
    )
    keep_linebreaks: bool = field(
        default=True, metadata={"help": "Whether to keep line breaks when using TXT files or not."}
    )
    dataset_path: Optional[str] = field(
        default='c4', metadata={"help": "The path of the dataset to use."}
    )

    disable_group_texts: bool = field(
        default=False,
        metadata={
            "help": (
                "Whether we disable group of original samples together to"
                " generate sample sequences of length `block_size`"
                " By Default, it is True, which means the long samples"
                " are truncated to `block_size` tokens"
                " and short samples are padded to `block_size` tokens."
                " If set to False, we group every 1000 tokenized"
                " sequences together, divide them into"
                " [{total_num_tokens} / {block_size}] sequences,"
                " each with `block_size` tokens"
                " (the remaining tokens are ommited."
                " This group text behavior is useful"
                " for continual pretrain or pretrain."
            )
        },
    )

    c4_ratio: Optional[float] = field(
        default=0.5,
        metadata={
            "help": (
                "For debugging purposes or quicker training, truncate the number of training examples to this "
                "value if set."
            )
        },
    )
    data_used: Optional[str] = field(
        default='only_c4', metadata={"help": "The path of the dataset to use."}
    )
    data_lr_scheduler: Optional[str] = field(
        default='cosine', metadata={"help": "The path of the dataset to use."}
    )
    merge4sft: str2bool = field(
        default=False,
        metadata={
            "help": (
                "Will use the token generated when running `huggingface-cli login` (necessary to use this script "
                "with private models)."
            )
        },
    )
    set_proportion: float = field(
        default=0.0,
        metadata={
            "help": (
                "Will use the token generated when running `huggingface-cli login` (necessary to use this script "
                "with private models)."
            )
        },
    )

    def __post_init__(self):
        if self.streaming:
            require_version("datasets>=2.0.0", "The streaming feature requires `datasets>=2.0.0`")

        if self.dataset_name is None and self.train_file is None and self.validation_file is None:
            raise ValueError("Need either a dataset name or a training/validation file.")
        else:
            if self.train_file is not None:
                extension = self.train_file.split(".")[-1]
                assert extension in ["csv", "json", "txt"], "`train_file` should be a csv, a json or a txt file."
            if self.validation_file is not None:
                extension = self.validation_file.split(".")[-1]
                assert extension in ["csv", "json", "txt"], "`validation_file` should be a csv, a json or a txt file."

def main():
    # See all possible arguments in src/transformers/training_args.py
    # or by passing the --help flag to this script.
    # We now keep distinct sets of args, for a cleaner separation of concerns.

    parser = HfArgumentParser((ModelArguments, DataTrainingArguments, TrainingArguments))
    if len(sys.argv) == 2 and sys.argv[1].endswith(".json"):
        # If we pass only one argument to the script and it's the path to a json file,
        # let's parse it to get our arguments.
        model_args, data_args, training_args = parser.parse_json_file(json_file=os.path.abspath(sys.argv[1]))
    else:
        model_args, data_args, training_args = parser.parse_args_into_dataclasses()

    # Sending telemetry. Tracking the example usage helps us better allocate resources to maintain them. The
    # information sent is the one passed as arguments along with your Python/PyTorch versions.
    send_example_telemetry("run_clm", model_args, data_args)

    ################################################################################################################
    batch_size = 128
    training_args.gradient_accumulation_steps = batch_size // training_args.per_device_train_batch_size
    if model_args.peft_type == "lora":
        training_args.warmup_steps = 5
    # training_args.fp16 = True
    training_args.logging_steps = 1
    training_args.optim = "adamw_torch"
    training_args.save_strategy = "no"
    training_args.eval_steps = 10
    #training_args.save_steps = 50
    training_args.save_total_limit = 1
    training_args.group_by_length = False

    # training_args.lr_scheduler_type = 'linear'
    training_args.num_train_epochs = int(training_args.num_train_epochs)
    training_args.max_train_steps = None
    # training_args.warmup_ratio = 0.03

    training_args.sparsity_ratio = model_args.sparsity_ratio
    training_args.merge_ratio = model_args.merge_ratio
    ################################################################################################################

    # If limit on cuda memory is specified enforce the limit
    if model_args.max_memory > 0:
        mem_info = torch.cuda.mem_get_info()
        print("Memory info: \n{}".format(mem_info))
        # total_memory = mem_info[1] * 1e-9 # convert Bytes to GB
        total_memory = torch.cuda.get_device_properties(0).total_memory * 2 ** (-30)  # convert Bytes to GB
        if model_args.max_memory > total_memory:
            raise ValueError(
                "The specified memory limit {} is greater than the available memory {}.".format(model_args.max_memory,
                                                                                                total_memory))
        else:
            fraction = model_args.max_memory / total_memory
            torch.cuda.set_per_process_memory_fraction(fraction)
            print("Restricting the memory to {} of the total memory to have a limit of {} ({} x {})".format(fraction,
                                                                                                            model_args.max_memory,
                                                                                                            fraction,
                                                                                                            total_memory))


    # Setup logging
    logging.basicConfig(
        format="%(asctime)s - %(levelname)s - %(name)s - %(message)s",
        datefmt="%m/%d/%Y %H:%M:%S",
        handlers=[logging.StreamHandler(sys.stdout)],
    )

    if training_args.should_log:
        # The default of training_args.log_level is passive, so we set log level at info here to have that default.
        transformers.utils.logging.set_verbosity_info()

    log_level = training_args.get_process_log_level()
    logger.setLevel(log_level)
    datasets.utils.logging.set_verbosity(log_level)
    transformers.utils.logging.set_verbosity(log_level)
    transformers.utils.logging.enable_default_handler()
    transformers.utils.logging.enable_explicit_format()

    # Log on each process the small summary:
    logger.warning(
        f"Process rank: {training_args.local_rank}, device: {training_args.device}, n_gpu: {training_args.n_gpu}"
        + f"distributed training: {bool(training_args.local_rank != -1)}, 16-bits training: {training_args.fp16}"
    )
    logger.info(f"Training/evaluation parameters {training_args}")

    # Detecting last checkpoint.
    last_checkpoint = None
    if os.path.isdir(training_args.output_dir) and training_args.do_train and not training_args.overwrite_output_dir:
        last_checkpoint = get_last_checkpoint(training_args.output_dir)
        if last_checkpoint is None and len(os.listdir(training_args.output_dir)) > 0:
            raise ValueError(
                f"Output directory ({training_args.output_dir}) already exists and is not empty. "
                "Use --overwrite_output_dir to overcome."
            )
        elif last_checkpoint is not None and training_args.resume_from_checkpoint is None:
            logger.info(
                f"Checkpoint detected, resuming training at {last_checkpoint}. To avoid this behavior, change "
                "the `--output_dir` or add `--overwrite_output_dir` to train from scratch."
            )

    # Set seed before initializing model.
    set_seed(training_args.seed)

    config_kwargs = {
        "cache_dir": model_args.cache_dir,
        "revision": model_args.model_revision,
        "use_auth_token": True if model_args.use_auth_token else None,
    }
    if model_args.config_name is None:
        model_args.config_name = model_args.model_name_or_path
    config = AutoConfig.from_pretrained(model_args.config_name, **config_kwargs)

    tokenizer_kwargs = {
        "cache_dir": model_args.cache_dir,
        # "use_fast": model_args.use_fast_tokenizer,
        "use_fast": False,
        "revision": model_args.model_revision,
        "trust_remote_code": True,
        "use_auth_token": True if model_args.use_auth_token else None,
    }
    # tokenizer = AutoTokenizer.from_pretrained(model_args.config_name, use_fast=False)

    # we use the tokenizer from vicuna
    # if "decapoda-research" in model_args.config_name:
    '''
    tokenizer = AutoTokenizer.from_pretrained(
        "lmsys/vicuna-13b-delta-v0",
        cache_dir=model_args.cache_dir,
        padding_side="right",
        use_fast=True,
    )
    '''
    ## "deepseek-ai/deepseek-llm-7b-chat"
    if 'deepseek' in model_args.model_name_or_path:
        tokenizer = AutoTokenizer.from_pretrained("deepseek-ai/deepseek-llm-7b-chat", **tokenizer_kwargs)
    else:
        tokenizer = AutoTokenizer.from_pretrained(model_args.model_name_or_path, **tokenizer_kwargs)

    if model_args.use_qpeft:
        model_kwargs = {}
        model_kwargs['quantization_config'] = BitsAndBytesConfig(
            load_in_4bit=True,
            bnb_4bit_use_double_quant=True,
            bnb_4bit_quant_type="nf4",
            bnb_4bit_compute_dtype=torch.bfloat16,
        )

        model = AutoModelForCausalLM.from_pretrained(
            model_args.model_name_or_path,
            low_cpu_mem_usage=True,
            # use_flash_attention_2=True,
            trust_remote_code=True,
            device_map="auto",
            **model_kwargs
        )

    else:
        model = AutoModelForCausalLM.from_pretrained(
            model_args.model_name_or_path,
            torch_dtype=torch.bfloat16,
            #cache_dir=model_args.cache_dir,
            low_cpu_mem_usage=True,
            #use_flash_attention_2=True,
            trust_remote_code=True,
            device_map="auto",
        )

    seq_len = 2048
    hidden_size = 4096

    if training_args.gradient_checkpointing:
        model.enable_input_require_grads()
        model.gradient_checkpointing_enable()

        logger.info(f"enable gradient_checkpointing !!!")


    print("##### base mode #####")
    for n, p in model.named_parameters():
        logger.info(f"{n}: {p.size()}, {p.requires_grad}, {p.dtype}")

    ############################################################################################
    #model = prepare_model_for_int8_training(model)
    if model_args.peft_type == "lora":
        peft_config = LoraConfig(
            r=model_args.lora_r,
            lora_alpha=model_args.lora_alpha,
            target_modules=["q_proj","k_proj", "v_proj", "o_proj", "up_proj", "down_proj", "gate_proj"],
            #target_modules=["q_proj","v_proj"],
            lora_dropout=model_args.lora_dropout,
            bias="none",
            task_type="CAUSAL_LM",
        )

    elif model_args.peft_type == "vera":
        peft_config = VeraConfig(
            r=model_args.lora_r,
            lora_alpha=model_args.lora_alpha,
            target_modules=["q_proj","k_proj", "v_proj", "o_proj", "up_proj", "down_proj", "gate_proj"],
            #target_modules=["q_proj","v_proj"],
            lora_dropout=model_args.lora_dropout,
            bias="none",
            task_type="CAUSAL_LM",
        )

    elif model_args.peft_type == "sft":
        if model_args.sft_density is None and model_args.sft_num_deltas is None:
            raise ValueError(
                f'When using SFT, one of --sft_density or --sft_num_deltas '
                f'must be specified.'
            )
        '''
        if model_args.sft_num_deltas is None:
            total_params = sum(p.numel() for p in model.parameters())
            model_args.sft_num_deltas = int(total_params * model_args.sft_density)
        
        target_modules = [
            (n, m) for n, m in model.named_modules()
            if n.startswith(model.base_model_prefix) and isinstance(m, torch.nn.Linear)
        ]
                
        nnz = {
            n: m.weight.count_nonzero()
            for n, m in target_modules
        }
        for n, c in nnz.items():
            logger.info(f"{n} has {c} nonzeros")
        '''
        if model_args.target_modules_model:
            target_modules = model_args.target_modules_model.split()
        else:
            target_modules = None

        peft_config = SftConfig(
            # task_type=TaskType.CAUSAL_LM,
            inference_mode=False,
            target_modules=target_modules,
            # num_deltas=num_deltas,
            # dtype=args.peft_dtype,
            dst=model_args.sparse_alg,
            selection_algorithm=model_args.sft_selection,
            reselection_steps=model_args.sft_reselection_steps,
            selection_accumulation_steps=model_args.sft_selection_accumulation_steps,
            mask_zeros=model_args.mask_zeros,
            pruned=model_args.pruned,
            sel_merge=data_args.merge4sft,
            set_proportion=data_args.set_proportion,
            initial_reselection_rate=model_args.initial_reselection_rate
        )

        print("peft_config.target_modules:{}".format(peft_config.target_modules))
        target_modules = []
        for n, m in model.named_modules():
            module = model.get_submodule(n)
            base_check = n.startswith(model.base_model_prefix) and isinstance(module, torch.nn.Linear)

            if base_check:
                if not peft_config.target_modules:
                    target_module_found = True
                elif isinstance(peft_config.target_modules, str):
                    target_module_found = re.fullmatch(peft_config.target_modules, n)
                else:
                    target_module_found = (
                        any(
                            re.search(f"{target_key}", n) #re.match(f".*\.{target_key}$", n)
                            for target_key in peft_config.target_modules
                        ) or
                        any(
                            target_key == n
                            for target_key in peft_config.target_modules
                        )
                    )
                if target_module_found:
                    target_modules.append((n, m))
        #print("target_modules:{}".format(target_modules))

        if not peft_config.target_modules and False:
            save_path = os.path.join(model_args.model_name_or_path, 'owl_before_rejust_npy.npy')
            all_layer_ratio_before = np.load(save_path)

            if model_args.using_allocate == 'owl':
                all_layer_ratio = (all_layer_ratio_before - all_layer_ratio_before.min()) * (
                        1 / (all_layer_ratio_before.max() - all_layer_ratio_before.min()) * model_args.Lamda * 2
                )
                nnz_p = all_layer_ratio - np.mean(all_layer_ratio) + (1 - model_args.sparsity_ratio)

            elif model_args.using_allocate == 'owl_reverse':
                all_layer_ratio = (all_layer_ratio_before.max() - all_layer_ratio_before) * (
                        1 / (all_layer_ratio_before.max() - all_layer_ratio_before.min()) * model_args.Lamda * 2
                )
                nnz_p = all_layer_ratio - np.mean(all_layer_ratio) + (1 - model_args.sparsity_ratio)

            else:
                nnz_p = np.ones(len(all_layer_ratio_before))

            layers = model.model.layers
            layer_modules = [(name, module) for name, module in layers[0].named_modules() if isinstance(module, torch.nn.Linear)]
            logger.info(f"layer modules num: {len(layer_modules)}, nnz_p:{nnz_p}")

            layer_num = len(layer_modules)
            nnz = {}
            for idx, (name, module) in enumerate(target_modules):
                current_nnz = nnz_p[idx // layer_num]
                nnz[name] = current_nnz
                logger.info(f"{name} has {current_nnz} nonzeros, p={idx},{nnz_p[idx % layer_num]}")

        else:
            nnz_p = np.ones(len(target_modules))
            nnz = {}
            for idx, (name, module) in enumerate(target_modules):
                nnz[name] = nnz_p[idx]
                logger.info(f"{name} has {nnz[name]} nonzeros, p={idx},{nnz_p[idx]}")

        nz_sum = sum(nnz.values())

        if model_args.sft_density == 1.0:
            num_deltas = {
                n: int(module.weight.count_nonzero())
                for n, module in target_modules
            }

        else:
            num_deltas = {
                n: int(nnz[n] * model_args.sft_num_deltas / nz_sum)
                for n, _ in target_modules
            }
        '''
        num_deltas = {
            n: int(nnz[n] * model_args.sft_num_deltas / nz_sum)
            for n, _ in target_modules
        }
        '''
        for n, c in num_deltas.items():
            logger.info(f"Initialising {n} with {c} tunable params")

        peft_config.num_deltas = num_deltas

        if model_args.sft_density == 1.0:
            peft_config.dst = 'fix'

    if model_args.peft_type != "full" and model_args.peft_type != "galore":
        model = get_peft_model(model, peft_config)
        model.print_trainable_parameters()
    ############################################################################################


    # We resize the embeddings only when necessary to avoid index errors. If you are creating a model from scratch
    # on a small vocab and want a smaller embedding size, remove this test.
    embedding_size = model.get_input_embeddings().weight.shape[0]
    if len(tokenizer) > embedding_size:
        model.resize_token_embeddings(len(tokenizer))
    '''
    # Preprocessing the datasets.
    lm_dataset_train_sft, lm_dataset_eval_sft = get_dataset_sft(training_args, data_args, model, tokenizer, model_args, logger)
    lm_datasets_c4 = get_dataset_c4(training_args, data_args, model_args, tokenizer, logger)

    if training_args.do_train:

        train_dataset_sft = lm_dataset_train_sft.get_backend_dataset()
        train_dataset_c4 = lm_datasets_c4["train"]

        if data_args.max_train_samples is not None:
            max_train_samples = min(len(train_dataset_sft), 2000)
            train_dataset_sft = train_dataset_sft.select(range(max_train_samples))

            max_train_samples = min(len(train_dataset_c4), 2500)
            train_dataset_c4 = train_dataset_c4.select(range(max_train_samples))


    if training_args.do_eval:
        # if "validation" not in tokenized_datasets:
        #     raise ValueError("--do_eval requires a validation dataset")

        if 'c4' not in data_args.dataset_path:
            eval_dataset = lm_dataset_eval.get_backend_dataset()
        else:
            eval_dataset = lm_datasets["validation"]

        # eval_dataset = lm_datasets["validation"]
        if data_args.max_eval_samples is not None:
            max_eval_samples = min(len(eval_dataset), data_args.max_eval_samples)
            eval_dataset = eval_dataset.select(range(max_eval_samples))

        def preprocess_logits_for_metrics(logits, labels):
            if isinstance(logits, tuple):
                # Depending on the model and config, logits may contain extra tensors,
                # like past_key_values, but logits always come first
                logits = logits[0]
            return logits.argmax(dim=-1)

        metric = evaluate.load("accuracy")

        def compute_metrics(eval_preds):
            preds, labels = eval_preds
            # preds have the same shape as the labels, after the argmax(-1) has been calculated
            # by preprocess_logits_for_metrics but we need to shift the labels
            labels = labels[:, 1:].reshape(-1)
            preds = preds[:, :-1].reshape(-1)
            return metric.compute(predictions=preds, references=labels)
    '''

    # Preprocessing the datasets.
    cal_loader = None
    if data_args.dataset_path.endswith(".json"):
        lm_dataset_train_sft, lm_dataset_eval_sft = get_dataset_alpaca(training_args, data_args, tokenizer, model_args, logger)
        train_dataset_sft = lm_dataset_train_sft
    else:
        lm_dataset_train_sft, lm_dataset_eval_sft = get_dataset_sft(training_args, data_args, model, tokenizer, model_args, logger)
        train_dataset_sft = lm_dataset_train_sft.get_backend_dataset()

    # lm_dataset_train_sft, lm_dataset_eval_sft = get_dataset_sft(training_args, data_args, model, tokenizer, model_args, logger)
    lm_datasets_c4, cal_loader = get_dataset_c4(training_args, data_args, model_args, tokenizer, logger, seq_len)

    # train_dataset_sft = lm_dataset_train_sft.get_backend_dataset()
    train_dataset_c4 = lm_datasets_c4["train"]

    #train_dataset_c4, cal_loader = None, None

    def select_samples(dataset, num_samples):
        selected_indices = np.random.choice(len(dataset), num_samples, replace=True)
        return dataset.select(selected_indices)


    if data_args.data_used == 'only_c4':
        if data_args.max_train_samples > 0:
            train_dataset = select_samples(train_dataset_c4, data_args.max_train_samples)
        else:
            train_dataset = train_dataset_c4

        num_training_steps = math.ceil(len(train_dataset) / training_args.gradient_accumulation_steps)

        num_training_steps_c4 = int(num_training_steps/2)
        num_training_steps_sft = num_training_steps - num_training_steps_c4

    elif data_args.data_used == 'only_sft':
        if data_args.max_train_samples > 0:
            train_dataset = select_samples(train_dataset_sft, data_args.max_train_samples)
        else:
            train_dataset = train_dataset_sft

        num_training_steps = math.ceil(len(train_dataset) / training_args.gradient_accumulation_steps)

        num_training_steps_c4 = int(num_training_steps/2)
        num_training_steps_sft = num_training_steps - num_training_steps_c4

    elif data_args.data_used == 'c4_then_sft':
        max_train_samples_c4 = int(data_args.max_train_samples * data_args.c4_ratio)
        max_train_samples_sft = data_args.max_train_samples - max_train_samples_c4

        train_dataset_c4 = select_samples(train_dataset_c4, max_train_samples_c4)
        train_dataset_sft = select_samples(train_dataset_sft, max_train_samples_sft)

        num_training_steps_c4 = math.ceil(len(train_dataset_c4) / training_args.gradient_accumulation_steps)
        num_training_steps_sft = math.ceil(len(train_dataset_sft) / training_args.gradient_accumulation_steps)
        num_training_steps = num_training_steps_c4 + num_training_steps_sft

        train_dataset = train_dataset_c4

    elif data_args.data_used == 'c4_add_sft':
        max_train_samples_c4 = int(data_args.max_train_samples * data_args.c4_ratio)
        max_train_samples_sft = data_args.max_train_samples - max_train_samples_c4

        train_dataset_c4 = select_samples(train_dataset_c4, max_train_samples_c4)
        train_dataset_sft = select_samples(train_dataset_sft, max_train_samples_sft)

        num_training_steps_c4 = math.ceil(len(train_dataset_c4) / training_args.gradient_accumulation_steps)
        num_training_steps_sft = math.ceil(len(train_dataset_sft) / training_args.gradient_accumulation_steps)
        num_training_steps = num_training_steps_c4 + num_training_steps_sft

        train_dataset = CombinedDataset(train_dataset_c4, train_dataset_sft)


    # Optimizer
    optimizer_grouped_parameters = [
        {
            "params": [
                p for n, p in model.named_parameters() if p.requires_grad
            ],
            "weight_decay": training_args.weight_decay,
        },
    ]

    if model_args.peft_type == 'lora':
        from bitsandbytes.optim import AdamW
        optimizer = AdamW(
            optimizer_grouped_parameters,
            lr=training_args.learning_rate,
        )
    elif model_args.peft_type == 'sft':
        optimizer = SftAdamW(
                optimizer_grouped_parameters,
                lr=training_args.learning_rate,
                momentum_dtype=torch.float32,
            )
    elif model_args.peft_type == 'galore':
        from galore_torch import GaLoreAdamW

        # make parameters with "rank" to a single group, if param_name has "mlp" or "attn"
        galore_params = []
        target_modules_list = ["attn", "mlp"]
        for module_name, module in model.named_modules():
            if not isinstance(module, torch.nn.Linear):
                continue

            if not any(target_key in module_name for target_key in target_modules_list):
                continue
            
            print('enable GaLore for weights in module: ', module_name)
            galore_params.append(module.weight)
        id_galore_params = [id(p) for p in galore_params]
        # make parameters without "rank" to another group
        regular_params = [p for p in model.parameters() if id(p) not in id_galore_params]
        # then call galore_adamw
        param_groups = [{'params': regular_params}, 
                        {'params': galore_params, 'rank':  model_args.lora_r, 'update_proj_gap': 100, 'scale': 1.0, 'proj_type': 'std'}]
        
        # galore_optimizer = GaLoreAdamW8bit(param_groups, lr=training_args.learning_rate)
        galore_optimizer = GaLoreAdamW(param_groups, lr=training_args.learning_rate)
        optimizer = galore_optimizer

    else:
        optimizer = torch.optim.AdamW(
            optimizer_grouped_parameters,
            lr=training_args.learning_rate,
        )


    ## learning rate scheduler
    '''
    lr_scheduler = get_scheduler(
        name=training_args.lr_scheduler_type,
        optimizer=optimizer,
        num_training_steps=num_training_steps,
        num_warmup_steps=int(num_training_steps * training_args.warmup_ratio),
    )
    
    lr_scheduler = WarmupStableDrop(
        optimizer=optimizer,
        start_lr=training_args.learning_rate, 
        warmup_iter=int(num_training_steps * training_args.warmup_ratio),
        end_iter=num_training_steps,
        drop_iter=int(num_training_steps/2)
    )
    '''

    num_training_steps = training_args.num_train_epochs * num_training_steps

    if 'WSD' in data_args.data_lr_scheduler: # and data_args.data_used == 'c4_then_sft':
        lr_scheduler = WarmupStableDrop(
            optimizer=optimizer,
            start_lr=training_args.learning_rate,
            warmup_iter=training_args.warmup_steps,
            end_iter=num_training_steps,
            drop_iter=num_training_steps_sft
        )
    elif '2WSD' in data_args.data_lr_scheduler: # and data_args.data_used == 'c4_then_sft':
        lr_scheduler = DoubleWSD(
            optimizer=optimizer,
            start_lr=training_args.learning_rate,
            warmup_iter=training_args.warmup_steps,
            end_iter=num_training_steps,
            drop_iter=num_training_steps_sft
        )
    else:
        if model_args.peft_type == "sft":
            training_args.warmup_steps = 0
        
        lr_scheduler = get_scheduler(
                name=data_args.data_lr_scheduler,
                optimizer=optimizer,
                num_training_steps=num_training_steps,
                num_warmup_steps=training_args.warmup_steps,
            )

    for n, p in model.named_parameters():
        logger.info(f"{n}: {p.size()}, {p.requires_grad}, {p.dtype}")

    ############## code imported from alpaca-lora ###################
    model.config.use_cache = False

    # combined_train_dataset = CombinedDataset(train_dataset_c4, train_dataset_sft)

    # Initialize our Trainer
    trainer_cls = Trainer
    trainer_kwargs = {}
    if model_args.peft_type == "sft":
        trainer_cls = SftTrainer(trainer_cls)  # MyTrainer = Trainer or any subclass thereof
        trainer_kwargs["sft_config"] = peft_config
        trainer_kwargs["total_update_steps"] = num_training_steps
        trainer_kwargs["cal_data"] = cal_loader

    trainer = trainer_cls(
        model=model,
        args=training_args,
        train_dataset=train_dataset if training_args.do_train else None,
        eval_dataset=eval_dataset if training_args.do_eval else None,
        tokenizer=tokenizer,
        # Data collator will default to DataCollatorWithPadding, so we change it.
        data_collator=default_data_collator,
        # compute_metrics=compute_metrics if training_args.do_eval and not is_torch_tpu_available() else None,
        # preprocess_logits_for_metrics=preprocess_logits_for_metrics
        #if training_args.do_eval and not is_torch_tpu_available()
        #else None,
        optimizers=(optimizer, lr_scheduler),
        **trainer_kwargs,
    )

    print("##### base and peft mode #####")
    for n, p in model.named_parameters():
        logger.info(f"{n}: {p.size()}, {p.requires_grad}, {p.dtype}")

    ############## code imported from alpaca-lora ###################
    model.config.use_cache = False

    # Training
    # total_steps = num_training_steps  # Example total steps
    # train_dataset_steps = int(total_steps/2)  # Steps for the first dataset
    # switch_dataset_steps = total_steps - train_dataset_steps

    if training_args.do_train:
        checkpoint = None
        if training_args.resume_from_checkpoint is not None:
            checkpoint = training_args.resume_from_checkpoint
        elif last_checkpoint is not None:
            checkpoint = last_checkpoint
    
        trainer.args.max_steps = math.ceil(len(train_dataset) / training_args.gradient_accumulation_steps) * training_args.num_train_epochs
        train_result = trainer.train(resume_from_checkpoint=checkpoint)
        
        if data_args.data_used == 'c4_then_sft':
            #if data_args.merge4sft:
            #    trainer._selector.select_merge()

            trainer.train_dataset = train_dataset_sft
            trainer.args.max_steps = math.ceil(len(train_dataset_sft) / training_args.gradient_accumulation_steps) * training_args.num_train_epochs
            train_result = trainer.train()

        if True:
            #############################################################
            pretrain_path = os.path.join(training_args.output_dir, 'pretrain')
            if not os.path.exists(pretrain_path):
                os.makedirs(pretrain_path)
    
            model.eval()  # Switch to evaluation mode
            with torch.no_grad():
                if model_args.peft_type != "full" and model_args.peft_type != "galore":
                    model = model.merge_and_unload()
            # model = model.merge_and_unload()
            model.save_pretrained(training_args.output_dir)
            tokenizer.save_pretrained(training_args.output_dir)
            # model.base_model.save_pretrained(pretrain_path)
            #############################################################
        else:
            trainer.save_model()  # Saves the tokenizer too for easy upload

        metrics = train_result.metrics

        max_train_samples = (
            data_args.max_train_samples if data_args.max_train_samples is not None else len(train_dataset)
        )
        metrics["train_samples"] = min(max_train_samples, len(train_dataset))

        trainer.log_metrics("train", metrics)
        trainer.save_metrics("train", metrics)
        # trainer.save_state()

    # Evaluation
    if training_args.do_eval:
        logger.info("*** Evaluate ***")

        metrics = trainer.evaluate()

        max_eval_samples = data_args.max_eval_samples if data_args.max_eval_samples is not None else len(
            eval_dataset)
        metrics["eval_samples"] = min(max_eval_samples, len(eval_dataset))
        # try:
        #    perplexity = math.exp(metrics["eval_loss"])
        # except OverflowError:
        #    perplexity = float("inf")
        # metrics["perplexity"] = perplexity

        trainer.log_metrics("eval", metrics)
        trainer.save_metrics("eval", metrics)

    kwargs = {"finetuned_from": model_args.model_name_or_path, "tasks": "text-generation"}
    if data_args.dataset_name is not None:
        kwargs["dataset_tags"] = data_args.dataset_name
        if data_args.dataset_config_name is not None:
            kwargs["dataset_args"] = data_args.dataset_config_name
            kwargs["dataset"] = f"{data_args.dataset_name} {data_args.dataset_config_name}"
        else:
            kwargs["dataset"] = data_args.dataset_name

    if training_args.push_to_hub:
        trainer.push_to_hub(**kwargs)
    else:
        trainer.create_model_card(**kwargs)


def _mp_fn(index):
    # For xla_spawn (TPUs)
    main()


if __name__ == "__main__":
    main()
