
import torch
from datasets import get_gcl_dataset
from models import get_model
from utils.status import progress_bar
from utils.tb_logger import *
from utils.status import create_stash
from models.utils.continual_model import ContinualModel
from argparse import Namespace


def evaluate(model: ContinualModel, dataset, eval_ema=False) -> float:
    """
    Evaluates the final accuracy of the model.
    :param model: the model to be evaluated
    :param dataset: the GCL dataset at hand
    :return: a float value that indicates the accuracy
    """
    curr_model = model.net
    if eval_ema:
        print('setting evaluation model to EMA model')
        curr_model = model.ema_model
    curr_model.eval()
    correct, total = 0, 0
    while not dataset.test_over:
        inputs, labels = dataset.get_test_data()
        inputs, labels = inputs.to(model.device), labels.to(model.device)
        outputs = model(inputs)
        _, predicted = torch.max(outputs[0].data, 1)
        correct += torch.sum(predicted == labels).item()
        total += labels.shape[0]

    acc = correct / total * 100
    return acc


def train(args: Namespace):
    """
    The training process, including evaluations and loggers.
    :param model: the module to be trained
    :param dataset: the continual dataset at hand
    :param args: the arguments of the current execution
    """
    if args.csv_log:
        from utils.loggers import CsvLogger
    dataset = get_gcl_dataset(args)
    backbone = dataset.get_backbone()
    loss = dataset.get_loss()
    model = get_model(args, backbone, loss, dataset.get_transform())
    model.net.to(model.device)

    model_stash = create_stash(model, args, dataset)


    lst_ema_models = ['ema_model']
    ema_loggers = {}
    ema_results = {}
    ema_results_mask_classes = {}
    ema_task_perf_paths = {}

    for ema_model in lst_ema_models:
        if hasattr(model, ema_model):
            ema_results[ema_model], ema_results_mask_classes[ema_model] = [], []

    if args.tensorboard:
        tb_logger = TensorboardLogger(args, dataset.SETTING, model_stash)
        csv_logger = CsvLogger(dataset.SETTING, dataset.NAME, model.NAME, tb_logger.get_log_dir())
        for ema_model in lst_ema_models:
            if hasattr(model, ema_model):
                print('=' * 50)
                print(f'Creating Logger for {ema_model}')
                print('=' * 50)
                path = os.path.join(tb_logger.get_log_dir(), ema_model)
                if not os.path.exists(path):
                    os.makedirs(path)
                ema_loggers[ema_model] = CsvLogger(dataset.SETTING, dataset.NAME, model.NAME, path)
                ema_task_perf_paths[ema_model] = os.path.join(path, 'task_performance_{}.txt'.format(ema_model))
        # from utils.loggers import CsvLogger
        # csv_logger = CsvLogger(tb_logger.get_log_dir())

    model.net.train()
    epoch, i = 0, 0
    ssl_loss = 0
    while not dataset.train_over:
        inputs, labels, not_aug_inputs = dataset.get_train_data()
        inputs, labels = inputs.to(model.device), labels.to(model.device)
        not_aug_inputs = not_aug_inputs.to(model.device)
        if args.train_ssl and dataset.completed_rounds < 3:
            ssl_loss += model.ssl_observe(not_aug_inputs, labels)
        else:
            loss, loss_rot = model.observe(inputs, labels, not_aug_inputs)
            if args.tensorboard:
                tb_logger.log_loss_gcl(loss, i)
            progress_bar(i, dataset.LENGTH // args.batch_size, epoch, 'C', loss)

        i += 1


    if model.NAME == 'joint_gcl':
      model.end_task(dataset)

    acc = evaluate(model, dataset)
    print('Accuracy:', acc)

    for ema_model in lst_ema_models:
        if hasattr(model, ema_model):
            print('=' * 30)
            print(f'Evaluating {ema_model}')
            print('=' * 30)
            dataset = get_gcl_dataset(args)
            ema_accs = evaluate(model, dataset, eval_ema=True)
            print('Accuracy:', ema_accs)
            if args.csv_log:
                ema_loggers[ema_model].log(ema_accs)

    if args.csv_log:
        csv_logger.log(acc)
        csv_logger.write(vars(args))
    for ema_model in ema_loggers:
        ema_loggers[ema_model].write(vars(args))