import ast
import copy
import logging
import os
import pandas as pd

import matplotlib.pyplot as plt
import numpy as np
from astropy.io import ascii
from matplotlib.ticker import MaxNLocator

from GenericTools.PlotTools.radar_plots import ComplexRadar
from GenericTools.StayOrganizedTools.unzip import unzip_good_exps
from GenericTools.StayOrganizedTools.utils import timeStructured
from ariel_tests.language.utils import samples_and_grammar_labels_from_file
from ariel_tests.test_analysis.visualization import PlotEncodingGrammarClustering, PlotDecodingGrammarClustering, \
    PlotEncodingLengthClustering

logger = logging.getLogger(__name__)

CDIR = os.path.dirname(os.path.realpath(__file__))
DATADIR = os.path.join(CDIR, 'data')
EXPERIMENTS = os.path.join(CDIR, 'experiments')
GEXPERIMENTS = os.path.join(CDIR, 'good_experiments')

ds = unzip_good_exps(
    GEXPERIMENTS, EXPERIMENTS, exp_identifiers=['experiment'], unzip_what='all',
    except_files=['samples.npy', 'samples_and_sentence.txt', 'biased_encoded.npz', 'files']
)


def retreiveTextsToListsAndLatex(experiments_folder):
    directories = [d for d in os.listdir(experiments_folder) if 'experiment' in d]
    experiments_list = []
    for j, filename in enumerate(directories):
        experiment_list = []
        experimentPath = os.path.join(experiments_folder, filename + '/text')

        print('')
        print(experimentPath)
        # retreive interpolations
        interpolation_string = []
        print('')
        with open(experimentPath + '/interpolation.txt', 'rt') as f:
            i = 0
            for line in f.readlines():
                i += 1
                if i > 7 and i % 2 == 1:
                    interpolation_string.append(line[1:-1])
                    print(interpolation_string[-1])
                if i == 30: break

        # retreive 10 generations            
        generation_string = []
        print('')
        with open(experimentPath + '/samples_and_sentence.txt', 'rt') as f:
            i = 0
            for line in f.readlines():
                if 'sentence' in line:
                    i += 1
                    generation_string.append(line[10:-1])
                    print(generation_string[-1])
                    if i == 10: break

        # retreive algebra            
        print('')
        algebra_string = []
        with open(experimentPath + '/linear_algebra.txt', 'rt') as f:
            i = 0
            for line in f.readlines():
                if i in [8, 17, 27]:
                    # print(line[29:-1])
                    algebra_string.append(line[29:-1])
                    print(algebra_string[-1])
                i += 1

        # print(experimentPath)
        # print(os.listdir(experimentPath))
        # print(interpolation_string)

        if 'arithmetic' in experimentPath:
            nameMethod = 'arithmetic'
        elif 'vae' in experimentPath:
            nameMethod = 'VAE'
        elif 'ae' in experimentPath:
            nameMethod = 'AE'
        elif 'lmariel' in experimentPath:
            nameMethod = 'AriEL'
        elif 'transformer' in experimentPath:
            nameMethod = 'Transformer'
        else:
            raise Exception('A config file with an unexpected method was introduced! Check!')

        nameMethod = str(j) + ' ' + nameMethod

        experiment_list += [nameMethod] + ['']
        experiment_list += interpolation_string + ['']
        experiment_list += algebra_string + ['']
        experiment_list += generation_string + ['']
        print(experiment_list)
        print('\n\n')

        experiments_list.append(experiment_list)

    ascii.write(experiments_list, format='latex')


def retreiveConfigGoodExperiments(experiments_folder):
    logger.info('Check if the experiments are in the expected folder')

    logger.info(experiments_folder)

    logger.info('Loading the data')
    allConfig = {}

    directories = sorted([d for d in os.listdir(experiments_folder) if 'experiment' in d])
    for i, filename in enumerate(directories):

        isNumberAtBeginningOfFile = True
        try:
            isNumberAtBeginningOfFile = isinstance(int(filename[0]), int)
        except:
            pass

        if isNumberAtBeginningOfFile:

            experimentPath = os.path.join(experiments_folder, filename)

            d_string = ''
            with open(experimentPath + '/config.txt', 'rt') as f:
                for line in f.readlines():
                    d_string += line

            # print(d_string)
            d_string = d_string.replace('array', '')
            d_string = d_string.replace('..., ', '')
            # print(d_string)
            config = ast.literal_eval(d_string)

            print(config['MethodName'])
            nameMethod = ''

            # if config['LanugageModel'] == 'LM':
            #    nameMethod += '-LM'

            if 'arithmetic' in config['MethodName']:
                nameMethod = 'arithmetic'
            elif 'vae' in config['MethodName']:
                nameMethod = 'VAE' + nameMethod
            elif 'ae' in config['MethodName']:
                nameMethod = 'AE' + nameMethod
            elif 'transformer' in config['MethodName']:
                nameMethod = 'Transformer' + nameMethod
            elif 'lmariel' in config['MethodName']:
                nameMethod = 'AriEL' + nameMethod
            else:
                raise Exception('A config file with an unexpected method was introduced! Check!')

            nameMethod = str(i) + ' ' + nameMethod

            allConfig[nameMethod] = config

    return allConfig


def dataToPercentStringsForLatex(latexData):
    percentData = copy.copy(latexData)
    for i in range(len(percentData)):
        for j in range(len(percentData[0])):
            if percentData[i][j] > 0.1:
                # percentData[i][j] = '{:.1%}'.format(percentData[i][j])
                percentData[i][j] = '{:.2%}'.format(percentData[i][j])
            else:
                percentData[i][j] = '{:.2%}'.format(percentData[i][j])

            percentData[i][j] = percentData[i][j].replace('%', '\%')

    return percentData


def writeLatexTable(latexData, measuresToPlot, names_methods):
    copiedData = copy.deepcopy(latexData)

    if True:
        # add mean and standard deviation
        means = [np.mean(c) for c in copiedData]
        stds = [np.std(c) for c in copiedData]
        names_methods.append('mean')
        extendedCopiedData = copiedData
        extendedCopiedData = [c + [means[i]] for i, c in enumerate(extendedCopiedData)]
        names_methods.append('std')
        extendedCopiedData = [c + [stds[i]] for i, c in enumerate(extendedCopiedData)]
        copiedData = extendedCopiedData

    copiedData = dataToPercentStringsForLatex(copiedData)
    names_methods.append('ms')
    copiedData = [c + [c[-2][:-2] + ' \pm \ ' + c[-1]] for i, c in enumerate(copiedData)]
    dataAndTitle = [names_methods] + copiedData
    ascii.write(dataAndTitle, format='latex', names=['model'] + measuresToPlot)


def plot_interpolation_diversity(x, diversity):
    # diversity = [1, 1, 0.956, 0.8393333333, 0.7826666667, 0.7693333333, 0.7533333333, 0.746, 0.746, 0.746, 0.746, 0.746,            0.746, ]
    # x = [1, 2, 4, 6, 8, 10, 12, 14, 16, 32, 256, 512, 1024, ]

    fig = plt.figure()
    ax = fig.add_subplot(2, 1, 1)

    line, = ax.plot(x, diversity, color='blue', lw=2)
    plt.xlabel('latent dimension d')
    # plt.xlabel('$\frac{sin(x)}{x}$')
    plt.ylabel('interpolation diversity')

    ax.set_xscale('log')

    plt.savefig('interpolations.pdf', bbox_inches='tight')


def plot(men_means, women_means, names_methods, measures):
    ind = np.arange(len(men_means))  # the x locations for the groups
    width = 0.35  # the width of the bars

    fig, ax = plt.subplots()
    rects1 = ax.bar(ind - width / 2, men_means, width,
                    color='SkyBlue', label=measures[0])
    rects2 = ax.bar(ind + width / 2, women_means, width,
                    color='IndianRed', label=measures[1])

    # Add some text for labels, title and custom x-axis tick labels, etc.
    ax.set_ylabel('Scores')
    ax.set_title('Scores by group and gender')
    ax.set_xticks(ind)
    ax.set_xticklabels(names_methods)
    ax.legend(loc='lower right')

    return fig


def FlexibleBarPlot(data, names_methods, measuresToPlot, title_bars=None):
    measuresInConfig = ['coverageGrammar', 'coverageVocabulary', 'nbCorrectAndUniqueSentences', 'nbUniqueSentences',
                        'coverage-biased', 'coverage-unbiased', 'grammatical', 'semantic']
    measuresSimpler = ['grammatical', 'coverage\nvocabulary', 'validity', 'uniqueness',
                       'reconstruction\ngrounded', 'reconstruction\nungrounded', 'grammatical', 'semantic']

    nbModels = len(data[0])
    nbMeasures = len(data)
    ind = np.arange(nbModels)  # the x locations for the groups

    width = 1 / (nbMeasures + 1)  # the width of the bars

    variables = [measuresSimpler[measuresInConfig.index(variable)] for variable in measuresToPlot]

    # example cmaps: 'viridis', 'magma', 'Paired'
    # cmap = plt.cm.get_cmap('Paired', nbMeasures)
    fig, ax = plt.subplots(figsize=(5, 3.5))
    for i in range(len(variables)):
        ax.bar(ind + i * width, data[i], width,
               label=variables[i])

    # Add some text for labels, title and custom x-axis tick labels, etc.
    ax.set_ylabel('Scores', fontsize=12)

    if not title_bars == None:
        ax.set_title(title_bars, fontsize=15)

    ax.set_xticks(ind + width * (nbMeasures - 1) / 2)
    ax.set_xticklabels(names_methods, fontsize=12)
    ax.legend(loc='right')

    plt.show()

    return fig


def FlexibleBarPlot2(data, names_methods, measuresToPlot, title_bars=None):
    measuresInConfig = ['coverageGrammar', 'coverageVocabulary', 'nbCorrectAndUniqueSentences', 'nbUniqueSentences',
                        'coverage-biased', 'coverage-unbiased', 'grammatical', 'semantic']
    measuresSimpler = ['grammatical', 'coverage\nvocabulary', 'validity', 'uniqueness',
                       'reconstruction\ngrounded', 'reconstruction\nungrounded', 'grammatical', 'semantic']

    measuresSimpler = ['GC', 'VC', 'V', 'U',
                       'RAG', 'RAU', 'GA', 'SA']

    nbModels = len(data[0])
    nbMeasures = len(data)
    ind = np.arange(nbMeasures)  # the x locations for the groups

    width = .9  # the width of the bars

    variables = [measuresSimpler[measuresInConfig.index(variable)] for variable in measuresToPlot]
    data = list(zip(*data))

    # example cmaps: 'viridis', 'magma', 'Paired'
    # cmap = plt.cm.get_cmap('Paired', nbMeasures)
    fig, ax = plt.subplots(nbModels, figsize=(2, 5))
    for i in range(nbModels):
        ax[i].bar(ind, data[i], width,
                  label=variables[i], zorder=1)
        ax[i].axhline(y=1, color='gray', linestyle='--', zorder=0)
        name_method = names_methods[i]
        ax[i].set_ylabel(name_method, fontsize=12)

    # Add some text for labels, title and custom x-axis tick labels, etc.
    #

    # if not title_bars == None:
    #    ax.set_title(title_bars, fontsize=15)

    ax[-1].set_xticks(ind)
    ax[-1].set_xticklabels(variables)
    # ax.legend(loc='right')

    plt.show()

    return fig


def plotThreeBarCharts(figs):
    # expected figs to be a list of fig objects
    total_fig, axes = plt.subplots(1, len(figs), sharex='col', sharey='row')

    for ax, fig in zip(axes, figs):
        ax = fig.gca()

    total_fig.show()
    return total_fig


def configToDataList(allConfig, measuresToPlot):

    data = [[] for _ in range(len(measuresToPlot))]
    names_methods = []

    for key, config in sorted(allConfig.items()):
        df = pd.json_normalize(config, sep='_')
        nconfig = df.to_dict(orient='records')[0]

        names_methods.insert(0, key)
        for k, v in nconfig.items():
            kk = k.replace('results_', '').replace('encoding_', '').replace('generation_', '').replace('reconstruction_', '')
            if kk in measuresToPlot:
                idx = measuresToPlot.index(kk)
                data[idx].insert(0, v)

    return data, names_methods


def dataInRadarFormat(data, namesModels, measuresToPlot, modelToSpyder='arithmetic'):
    measuresInConfig = ['grammar-clustering-encoding', 'length-clustering-encoding',
                        'coverageGrammar', 'coverageVocabulary', 'nbCorrectAndUniqueSentences', 'nbUniqueSentences',
                        'coverage-biased', 'coverage-unbiased', 'grammatical', 'semantic']
    measuresSimpler = ['grammar\nEncoding', 'length\nEncoding',
                       'grammar\ncoverage\nGeneration', 'vocabulary\ncoverage\nGeneration', 'validity\nGeneration',
                       'uniqueness\nGeneration',
                       'reconstruction\naccuracy\nbiased\nReconstruction',
                       'reconstruction\naccuracy\nunbiased\nGeneralization', 'grammar\naccuracy\nReconstruction',
                       'semantic\naccuracy\nReconstruction']

    # make the data for a spyder chart with one model

    assert modelToSpyder in namesModels
    modelIdx = namesModels.index(modelToSpyder)
    data_T = list(map(list, zip(*data)))

    variables = [measuresSimpler[measuresInConfig.index(variable)] for variable in measuresToPlot]
    dataRadar = tuple(data_T[modelIdx])
    variables = tuple(variables)

    return dataRadar, variables


def FlexibleSpyderChart(data, namesModels, measuresToPlot, title):
    # plotting
    fig = plt.figure(figsize=(7, 5))

    handles = []
    _, variables = dataInRadarFormat(data, namesModels, measuresToPlot, namesModels[0])  # 'arithmetic')

    ranges = [(0, 1)] * len(variables)
    radar = ComplexRadar(fig, variables, ranges)

    for modelName in namesModels:
        dataSpyder, _ = dataInRadarFormat(data, namesModels, measuresToPlot, modelName)

        handle = radar.plot(dataSpyder, label=modelName)
        radar.fill(dataSpyder, alpha=0.1, label=modelName)

        handles.append(handle)

    ax = plt.gca()
    # title = r"N_d = " + title
    ax.set_title(title, y=1.02, x=1.2)
    # ax.legend(handles=handles, bbox_to_anchor=(1.05, 1), loc=2, borderaxespad=0.)
    plt.show()

    return fig


def plotQualitative45(experiments_folder, projection_method='PCA'):
    n_samples2plot = 10000  # 10000 #nbMaxSentences_biased
    fontsize = 15

    # pre-compute to which grammar rule each Test sentence belongs to
    nbMaxSentences_biased = 10000

    filename = 'labels_grammar_biasedTest_HQ_%s.npy' % (nbMaxSentences_biased)
    path = os.path.join(DATADIR, filename)
    grammar_labels = np.load(path)
    filename = 'labels_length_biasedTest_HQ_%s.npy' % (nbMaxSentences_biased)
    path = os.path.join(DATADIR, filename)
    length_labels = np.load(path)

    length_labels = length_labels[:n_samples2plot]
    grammar_labels = grammar_labels[:n_samples2plot]

    all_directories = [d for d in os.listdir(experiments_folder) if 'experiment' in d]
    all_directories = sorted(all_directories)

    # the following is to only take into account manually preselected experiments
    # that have been numerated
    directories = []
    for filename in all_directories:
        try:
            if isinstance(int(filename[0]), int):
                directories.append(filename)
        except:
            pass

    names_directories = []
    for d in directories:
        if 'ariel' in d:
            names_directories.append('AriEL')
        elif 'transformer' in d:
            names_directories.append('Transformer')
        elif 'vae' in d:
            names_directories.append('VAE')
        elif 'ae' in d:
            names_directories.append('AE')
        else:
            raise NotImplementedError

    bigFig, axes = plt.subplots(3, len(directories), sharex=True, sharey=True, figsize=(50 / 5.5, 30 / 5.5))

    if len(directories) == 1:
        axx = [axes[0]]
    else:
        axx = axes[0]

    if len(directories) == 1: axes = axes[:, np.newaxis]
    for ax, dir, col in zip(axx, directories, names_directories):
        ax.set_title(col, fontsize=fontsize)

    # pre-compute to which grammar rule each Test sentence belongs to

    gEnc_axes = axes[1, :]
    for experiment_folder, ax in zip(directories, gEnc_axes):
        d = experiments_folder + '/' + experiment_folder + '/text/biased_encoded.npz'
        data = np.load(d)
        encoded_sentences = data['arr_0']
        encoded_sentences = encoded_sentences[:n_samples2plot]

        if 'arithmetic' in experiment_folder:
            dims_chosen = [0, 2]
        elif 'vae' in experiment_folder:
            # vae doesn't seem to have a better clustering in any direction
            dims_chosen = [0, 2]
        elif 'ae' in experiment_folder:
            dims_chosen = [1, 2]
        elif 'transformer' in experiment_folder:
            dims_chosen = [0, 1]
        elif 'lmariel' in experiment_folder:
            dims_chosen = [0, 1]
        else:
            raise NotImplementedError

        PlotEncodingGrammarClustering(
            DataClass=None,
            encoded_sentences=encoded_sentences,
            grammar_labels=grammar_labels,
            experiments_folder=experiments_folder,
            experiment_folder=experiment_folder,
            ax_out=ax,
            dims_chosen=dims_chosen,
            projection_method=projection_method)
        ax.set(aspect='equal')

    # ax.legend(loc='upper center', bbox_to_anchor=(1.45, 0.8), ncol=1)

    lEnc_axes = axes[2, :]
    for experiment_folder, ax in zip(directories, lEnc_axes):
        d = experiments_folder + '/' + experiment_folder + '/text/biased_encoded.npz'
        data = np.load(d)
        encoded_sentences = data['arr_0']
        encoded_sentences = encoded_sentences[:n_samples2plot]

        if 'arithmetic' in experiment_folder:
            dims_chosen = [0, 2]
        elif 'vae' in experiment_folder:
            # vae doesn't seem to have a better clustering in any direction
            dims_chosen = [0, 2]
        elif 'ae' in experiment_folder:
            dims_chosen = [1, 2]
        elif 'transformer' in experiment_folder:
            dims_chosen = [0, 1]
        elif 'lmariel' in experiment_folder:
            dims_chosen = [0, 1]
        else:
            raise NotImplementedError

        PlotEncodingLengthClustering(
            DataClass=None,
            encoded_sentences=encoded_sentences,
            length_labels=length_labels,
            experiments_folder=experiments_folder,
            experiment_folder=experiment_folder,
            ax_out=ax,
            dims_chosen=dims_chosen,
            projection_method=projection_method)
        ax.set(aspect='equal')

    gDec_axes = axes[0, :]
    for experiment_folder, ax in zip(directories, gDec_axes):
        d_samples = experiments_folder + experiment_folder + '/text/samples.npy'
        d_grammar_labels = experiments_folder + experiment_folder + '/text/samples_grammar_labels.npy'

        elementsFilename = experiment_folder.split('_')
        latDim = int([e[:-1] for e in elementsFilename if e[-1] == 'd'][0])

        if os.path.isfile(d_samples):
            samples = np.load(d_samples)
            grammar_labels = np.load(d_grammar_labels)

            print('samples and labels were successfully loaded')

        else:
            text_path = experiments_folder + '/' + experiment_folder + '/text/samples_and_sentence.txt'
            samples, grammar_labels = samples_and_grammar_labels_from_file(text_path)
            print('samples and labels were not saved, computing them from files and saving them')

            np.save(experiments_folder + '/' + experiment_folder + '/text/samples.npy', samples)
            np.save(experiments_folder + '/' + experiment_folder + '/text/samples_grammar_labels.npy', grammar_labels)

        if 'arithmetic' in experiment_folder:
            dims_chosen = [0, 2]
        elif 'vae' in experiment_folder:
            # vae doesn't seem to have a better clustering in any direction
            dims_chosen = [12, 0]
        elif 'ae' in experiment_folder:
            dims_chosen = [13, 5]  # [8,3]
        elif 'transformer' in experiment_folder:
            dims_chosen = [0, 1]
        elif 'lmariel' in experiment_folder:
            dims_chosen = [6, 3]  # [3,13]
        else:
            raise NotImplementedError

        samples = samples[:n_samples2plot]
        grammar_labels = grammar_labels[:n_samples2plot]

        PlotDecodingGrammarClustering(
            DataClass=None,
            samples=samples,
            labels=grammar_labels,
            lat_dim=latDim,
            experiments_folder=experiments_folder,
            experiment_folder=experiment_folder,
            ax_out=ax,
            dims_chosen=dims_chosen)
        ax.set(aspect='equal')

    for ax in bigFig.axes:
        # axes numbers to integers
        locator = MaxNLocator(prune='both', nbins=1, integer=True)
        ax.yaxis.set_major_locator(locator)
        ax.xaxis.set_major_locator(locator)

    # grammar legend  
    lgnd = axes[0, 2].legend(loc='center left', bbox_to_anchor=(-1.7, -2.6), ncol=1, fontsize=fontsize)
    for l in lgnd.legendHandles: l._sizes = [40]

    # length legend
    lgnd = axes[2, 2].legend(loc='center left', bbox_to_anchor=(-.0, -.6), ncol=1, fontsize=fontsize)
    for l in lgnd.legendHandles: l._sizes = [40]

    bigFig.subplots_adjust(top=1, bottom=0, right=1, left=0,
                           hspace=0, wspace=-0.51)
    idx = experiments_folder.index('experiments')

    time_string = timeStructured()
    plot_path = os.path.join(experiments_folder[:idx],
                             'experiments/good_models/one_of_each_16/plots/{}_qualitative45'.format(time_string))
    bigFig.savefig(plot_path + '.pdf', bbox_inches="tight")
    bigFig.savefig(plot_path + '.png', bbox_inches="tight", dpi=1200)


def main():

    plots_folder = os.path.join(EXPERIMENTS, 'plots') #+ '/plots/'
    if not os.path.isdir(plots_folder): os.mkdir(plots_folder)

    # studies to perform
    qualitative123 = False
    qualitative45 = False
    isWriteLatexTable = True
    radarPlots = False
    barPlots = False

    quantitative = any([isWriteLatexTable, radarPlots, barPlots])

    if qualitative123:
        retreiveTextsToListsAndLatex(EXPERIMENTS)

    if qualitative45:
        plotQualitative45(EXPERIMENTS)

    if quantitative:
        allConfig = retreiveConfigGoodExperiments(EXPERIMENTS)
        measuresToPlot = [  # 'grammar-clustering-encoding', 'length-clustering-encoding','semantic',
            'coverageGrammar', 'coverageVocabulary', 'nbCorrectAndUniqueSentences', 'nbUniqueSentences',
            'grammatical', 'coverage-biased', 'coverage-unbiased', 'memorized_ratio']
        print(allConfig)
        # prepare the data in a convenient way to be plot
        data, namesModels = configToDataList(allConfig, measuresToPlot)
        print(data)

    if isWriteLatexTable:
        # write the data in a latex table
        writeLatexTable(data, measuresToPlot, namesModels)

    if radarPlots:
        # plot different coverages in a radar chart    
        fig = FlexibleSpyderChart(data, namesModels, measuresToPlot, 'x units')
        fig.savefig(plots_folder + 'radar.pdf', bbox_inches='tight')

    if barPlots:
        title_bars = 'Generation'
        # measuresToPlot = ['coverageGrammar', 'coverageVocabulary', 'nbCorrectAndUniqueSentences', 'nbUniqueSentences']

        # title_bars = 'Generalization'
        # measuresToPlot = ['coverage-biased', 'coverage-unbiased']

        # title_bars = 'Reconstruction'
        # measuresToPlot = ['coverage-biased', 'grammatical', 'semantic']

        # plot the measures chosen in a bar plot
        # fig = FlexibleBarPlot(data, namesModels, measuresToPlot, title_bars)
        fig = FlexibleBarPlot2(data, namesModels, measuresToPlot, title_bars)
        fig.savefig(plots_folder + str(len(data)) + '_attempt.pdf', bbox_inches='tight')

        # title_bars_1 = 'Generation'
        # measuresToPlot_1 = ['coverageGrammar', 'coverageVocabulary', 'nbCorrectAndUniqueSentences', 'nbUniqueSentences']

        # title_bars_2 = 'Reconstruction'
        # measuresToPlot_2 = ['coverage-biased', 'coverage-unbiased']

        # title_bars_3 = 'Reconstruction'
        # measuresToPlot_3 = ['coverage-biased', 'grammatical', 'semantic']

        # three plots together
        # tot_titles = [title_bars_3, title_bars_2, title_bars_3]
        # tot_measuresToPlot = [measuresToPlot_1, measuresToPlot_2, measuresToPlot_3]

        # figures = []
        # for title, measures in zip(tot_titles, tot_measuresToPlot):
        # prepare the data in a convenient way to be plot
        #    data, namesModels = configToDataList(allConfig, measures)
        #    fig = FlexibleBarPlot(data, namesModels, measuresToPlot_3, title_bars_3)
        #    figures.append(fig)

        # plotThreeBarCharts(figures)


if __name__ == '__main__':
    main()
