import torch
import torch.nn as nn
import torch.nn.functional as F

from models.layers import *


class BasicBlock(nn.Module):
    def __init__(self, T, in_planes, out_planes, stride, dropRate=0.0 ):
        super(BasicBlock, self).__init__()
        self.T = T
        self.bn1 = tdBatchNorm(in_planes)
        self.act1 = LIFSpike()
        self.conv1 = SeqToANNContainer(nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride, padding=1, bias=False))
        self.bn2 = tdBatchNorm(out_planes)
        self.act2 = LIFSpike()
        self.conv2 = SeqToANNContainer(nn.Conv2d(out_planes, out_planes, kernel_size=3, stride=1, padding=1, bias=False))
        self.droprate = dropRate
        self.equalInOut = (in_planes == out_planes)
        self.convShortcut = (not self.equalInOut) and SeqToANNContainer(nn.Conv2d(in_planes, out_planes, kernel_size=1, stride=stride, padding=0, bias=False)) or None
        self.convex = ConvexCombination(2)
        

    def forward(self, x):
        if not self.equalInOut:
            x,v1 = self.act1(self.bn1(x))
        else:
            out,v1 = self.act1(self.bn1(x))
        out,v2 = self.act2(self.bn2(self.conv1(out if self.equalInOut else x)))
        if self.droprate > 0:
            out = F.dropout(out, p=self.droprate, training=self.training)
        out = self.conv2(out)
        return self.convex(x if self.equalInOut else self.convShortcut(x), out), v1+v2

class wrn16(nn.Module):
    def __init__(self, num_classes, norm, dropRate=0.0):
        super(wrn16, self).__init__()
        depth = 16
        widen_factor = 4
        nChannels = [16, 16*widen_factor, 32*widen_factor, 64*widen_factor]
        n = 2

        if norm is not None and isinstance(norm, tuple):
            self.norm = TensorNormalization(*norm)
        else:
            raise AssertionError("Invalid normalization")

        block = BasicBlock
        self.T = 4

        # 1st conv before any network block
        self.conv1 = SeqToANNContainer(nn.Conv2d(3, nChannels[0], kernel_size=3, stride=1, padding=1, bias=False))

        self.block1 = self._make_layer(block, nChannels[0], nChannels[1], n, 1, dropRate)
        self.block2 = self._make_layer(block, nChannels[1], nChannels[2], n, 2, dropRate)
        self.block3 = self._make_layer(block, nChannels[2], nChannels[3], n, 2, dropRate)

        self.bn1 = tdBatchNorm(nChannels[3])
        self.act = LIFSpike()
        self.avgpool = tdLayer(nn.AvgPool2d(kernel_size=8))
        self.flatten = nn.Flatten(2)
        self.fc = SeqToANNContainer(nn.Linear(nChannels[3], num_classes))
        self.nChannels = nChannels[3]

        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
            elif isinstance(m, nn.BatchNorm2d):
                m.weight.data.fill_(1)
                m.bias.data.zero_()
            elif isinstance(m, nn.Linear):
                m.bias.data.zero_()
    
    def _make_layer(self, block, in_planes, out_planes, nb_layers, stride, dropRate):
        layers = []
        for i in range(int(nb_layers)):
            layers.append(block(self.T, i == 0 and in_planes or out_planes, out_planes, i == 0 and stride or 1, dropRate))
        return nn.Sequential(*layers)


    def forward(self, x):
        x = self.norm(x)
        x = add_dimention(x, self.T)
        v = 0

        x = self.conv1(x)

        for i in range(len(self.block1)):
            x, v1 = self.block1[i](x)
            v = v + v1

        for i in range(len(self.block2)):
            x, v1 = self.block2[i](x)
            v = v + v1

        for i in range(len(self.block3)):
            x, v1 = self.block3[i](x)
            v = v + v1

        x, v1 = self.act(self.bn1(x))
        v = v + v1

        x = self.avgpool(x)

        x = self.flatten(x)

        x = self.fc(x)

        return x,v
