# Minimal Reproduction Guide for Running OrthogonalMatchingPursuit

This guide provides step-by-step instructions to reproduce the functionality of the `OrthogonalMatchingPursuit` class from `dmi.py`, including setting up the kernel matrix and scoring embedding matrix.

## Steps to Reproduce

### 1. Prepare Input Data

- **Kernel Matrix**: A square matrix representing pairwise similarities or relationships between items. For this example, we generate a random kernel matrix.
- **Scoring Embedding Matrix**: A matrix where each row represents the embedding of an item. This is used to compute scores during selection.

Example:
```python
import numpy as np

# Set random seed for reproducibility
np.random.seed(42)

# Define dimensions
N = 50  # Number of items
D = 128  # Embedding dimension

# Generate random kernel matrix (N x N)
kernel_matrix = np.random.rand(N, N).astype(np.float64)

# Generate random scoring embedding matrix (N x D)
emb_mat = np.random.rand(N, D).astype(np.float64)
```

### 2. Instantiate the `OrthogonalMatchingPursuit` Class

Import the `OrthogonalMatchingPursuit` class from `dmi.py` and create an instance, passing the argument `process_reward_args={'method': 'flat'}`. This corresponds to the MP + Self Compress method used in the paper:
```python
from dmi import OrthogonalMatchingPursuit

# Instantiate the class
omp = OrthogonalMatchingPursuit(
    reward_mat=kernel_matrix, 
    emb_mat=emb_mat, 
    process_reward_args={'method': 'flat'}
)
```

### 3. Run the Selection

Call the `compute_selection` method with the desired maximum number of items to select (`max_length`):
```python
# Define the maximum number of items to select
max_length = 10

# Run the selection
selected_items = omp.compute_selection(max_length)
print("Selected items:", selected_items)
```

### Complete Example

Here is the complete code for minimal reproduction:
```python
import numpy as np
from dmi import OrthogonalMatchingPursuit

# Set random seed for reproducibility
np.random.seed(42)

# Define dimensions
N = 50  # Number of items
D = 128  # Embedding dimension
max_length = 10  # Number of items to select

# Generate random kernel matrix (N x N)
kernel_matrix = np.random.rand(N, N).astype(np.float64)

# Generate random scoring embedding matrix (N x D)
emb_mat = np.random.rand(N, D).astype(np.float64)

# Instantiate the OrthogonalMatchingPursuit class
omp = OrthogonalMatchingPursuit(
    reward_mat=kernel_matrix, 
    emb_mat=emb_mat, 
    process_reward_args={'method': 'flat'}
)

# Run the selection
selected_items = omp.compute_selection(max_length)
print("Selected items:", selected_items)
```

## Notes

- Ensure the kernel matrix and embedding matrix are properly normalized if required by your use case.
- The `compute_selection` method selects items based on a greedy update strategy using the provided matrices.
- This example uses random data for demonstration purposes. Replace it with your actual data for real-world use cases.
- The `process_reward_args={'method': 'flat'}` argument enables the MP + Self Compress method as described in the paper(MP+SC).
