#pragma once

#include "police/macros.hpp"
#include "police/storage/id_map.hpp"
#include "police/storage/unordered_map.hpp"

#include <any>
#include <memory>
#include <string>
#include <type_traits>
#include <utility>

namespace police {

class ArgumentStore : public std::enable_shared_from_this<ArgumentStore> {
public:
    template <typename T>
    void set(const std::string& key, T&& value)
    {
        pos_.insert(key);
        auto it = value_store_.find(key);
        auto ptr =
            std::make_shared<std::remove_cv_t<T>>(std::forward<T>(value));
        if (it == value_store_.end()) {
            value_store_.emplace(key, std::move(ptr));
        } else {
            it->second = std::move(ptr);
        }
    }

    template <typename T>
    void set_ptr(const std::string& key, std::shared_ptr<T> ptr)
    {
        set(key, std::move(ptr));
    }

    template <typename T>
    [[nodiscard]]
    T pop(const std::string& key)
    {
        auto ptr = find<T>(key);
        remove(key);
        return std::move(*ptr);
    }

    template <typename T>
    [[nodiscard]]
    const T& get(const std::string& key) const
    {
        return *find<T>(key);
    }

    template <typename E>
    [[nodiscard]]
    const E get_enum(const std::string& key) const
    {
        return E(get<int>(key));
    }

    template <typename T>
    [[nodiscard]]
    const std::shared_ptr<T> get_ptr(const std::string& key) const
    {
        return get<std::shared_ptr<T>>(key);
    }

    [[nodiscard]]
    bool has(const std::string& key) const
    {
        return value_store_.count(key);
    }

    void remove(const std::string& key)
    {
        auto it = value_store_.find(key);
        if (it != value_store_.end()) {
            pos_.erase(pos_.find(key));
            value_store_.erase(it);
        }
    }

    template <typename T>
    void store_persistently(std::shared_ptr<T> obj)
    {
        objs_.push_back(std::move(obj));
    }

private:
    template <typename T>
    std::shared_ptr<T> find(const std::string& key) const
    {
        auto val = value_store_.find(key);
        if (val == value_store_.end()) {
            POLICE_RUNTIME_ERROR("There is no value for argument " << key);
        }
        try {
            return std::any_cast<std::shared_ptr<T>>(val->second);
        } catch (const std::bad_any_cast&) {
            POLICE_RUNTIME_ERROR(
                "Type mismatch while retrieving value of "
                "argument "
                << key << " of type " << typeid(T).name());
        }
    }

    id_map<std::string> pos_;
    unordered_map<std::string, std::any> value_store_;
    vector<std::any> objs_;
};

} // namespace police
