#include "police/expressions/boolean_combination.hpp"
#include "police/expressions/expression.hpp"
#include "police/expressions/expression_transformer.hpp"
#include "police/expressions/expression_visitor.hpp"
#include "police/utils/hash.hpp"
#include <algorithm>
#include <functional>
#include <numeric>

namespace police::expressions {

void BooleanCombination::dump_(std::ostream& out, std::string_view connector)
    const
{
    if (children.empty()) {
        out << "(" << connector << ")";
    } else {
        out << "(" << children.front();
        std::for_each(children.begin() + 1, children.end(), [&](auto&& e) {
            out << " " << connector << " " << e;
        });
        out << ")";
    }
}

bool Conjunction::operator==(const Conjunction& other) const
{
    return children.size() == other.children.size() &&
           std::transform_reduce(
               children.begin(),
               children.end(),
               other.children.begin(),
               true,
               std::logical_and<>(),
               [](const auto& e1, const auto& e2) { return e1.is_same(e2); });
}

std::size_t Conjunction::hash() const
{
    return hash_combine(typeid(Conjunction).hash_code(), get_hash(children));
}

void Conjunction::dump(std::ostream& out) const
{
    dump_(out, "&&");
}

bool Disjunction::operator==(const Disjunction& other) const
{
    return children.size() == other.children.size() &&
           std::transform_reduce(
               children.begin(),
               children.end(),
               other.children.begin(),
               true,
               std::logical_and<>(),
               [](const auto& e1, const auto& e2) { return e1.is_same(e2); });
}

void Disjunction::dump(std::ostream& out) const
{
    dump_(out, "||");
}

std::size_t Disjunction::hash() const
{
    return hash_combine(typeid(Disjunction).hash_code(), get_hash(children));
}

__POLICE_IMPLEMENT_EXPR_VISITORS(Conjunction)
__POLICE_IMPLEMENT_EXPR_VISITORS(Disjunction)
} // namespace police::expressions
