#include "police/expressions/expression.hpp"
#include "police/expressions/binary_function_call.hpp"
#include "police/expressions/comparison.hpp"
#include "police/expressions/expressions.hpp"
#include "police/expressions/ifthenelse.hpp"
#include "police/macros.hpp"

#include <memory>

namespace police::expressions {

bool RawExpression::is_constant() const
{
    return false;
}

Value RawExpression::get_value() const
{
    POLICE_RUNTIME_ERROR(
        "attempted accessing constant value of non-constant expression");
}

Expression::Expression(Value value)
    : expr_(std::make_shared<Constant>(std::move(value)))
{
}

Expression::Expression(identifier_name_t name)
    : expr_(std::make_shared<IdentifierReference>(std::move(name)))
{
}

void Expression::accept(ExpressionVisitor& visitor) const
{
    expr_->accept(visitor);
}

void Expression::transform(ExpressionTransformer& transform)
{
    expr_->transform(*this, transform);
}

bool Expression::is_same(const Expression& other) const
{
    return expr_->is_same(other);
}

bool Expression::is_constant() const
{
    return expr_->is_constant();
}

Value Expression::get_value() const
{
    return expr_->get_value();
}

Expression operator+(Expression left, Expression right)
{
    return NumericOperation(
        NumericOperation::Operand::ADD,
        std::move(left),
        std::move(right));
}

Expression operator-(Expression left, Expression right)
{
    return NumericOperation(
        NumericOperation::Operand::SUBTRACT,
        std::move(left),
        std::move(right));
}

Expression operator*(Expression left, Expression right)
{
    return NumericOperation(
        NumericOperation::Operand::MULTIPLY,
        std::move(left),
        std::move(right));
}

Expression operator/(Expression left, Expression right)
{
    return NumericOperation(
        NumericOperation::Operand::DIVISION,
        std::move(left),
        std::move(right));
}

Expression operator%(Expression left, Expression right)
{
    return NumericOperation(
        NumericOperation::Operand::MODULO,
        std::move(left),
        std::move(right));
}

Expression operator&&(Expression left, Expression right)
{
    return Conjunction({std::move(left), std::move(right)});
}

Expression operator||(Expression left, Expression right)
{
    return Disjunction({std::move(left), std::move(right)});
}

Expression operator==(Expression left, Expression right)
{
    return equal(std::move(left), std::move(right));
}

Expression operator!=(Expression left, Expression right)
{
    return not_equal(std::move(left), std::move(right));
}

Expression operator<=(Expression left, Expression right)
{
    return less_equal(std::move(left), std::move(right));
}

Expression operator>=(Expression left, Expression right)
{
    return greater_equal(std::move(left), std::move(right));
}

Expression operator<(Expression left, Expression right)
{
    return less(std::move(left), std::move(right));
}

Expression operator>(Expression left, Expression right)
{
    return greater(std::move(left), std::move(right));
}

Expression equal(Expression left, Expression right)
{
    return Comparison(
        Comparison::Operator::EQUAL,
        std::move(left),
        std::move(right));
}

Expression not_equal(Expression left, Expression right)
{
    return Comparison(
        Comparison::Operator::NOT_EQUAL,
        std::move(left),
        std::move(right));
}

Expression less_equal(Expression left, Expression right)
{
    return Comparison(
        Comparison::Operator::LESS_EQUAL,
        std::move(left),
        std::move(right));
}

Expression greater_equal(Expression left, Expression right)
{
    return Comparison(
        Comparison::Operator::LESS_EQUAL,
        std::move(right),
        std::move(left));
}

Expression greater(Expression left, Expression right)
{
    return Comparison(
        Comparison::Operator::LESS,
        std::move(right),
        std::move(left));
}

Expression less(Expression left, Expression right)
{
    return Comparison(
        Comparison::Operator::LESS,
        std::move(left),
        std::move(right));
}

Expression operator!(Expression left)
{
    return Negation(std::move(left));
}

Expression ite(Expression cond, Expression then, Expression otherwise)
{
    return IfThenElse(std::move(cond), std::move(then), std::move(otherwise));
}

Expression min(Expression x, Expression y)
{
    return BinaryFunctionCallGeneric(
        BinaryFunctionCallGeneric::MIN,
        std::move(x),
        std::move(y));
}

Expression max(Expression x, Expression y)
{
    return BinaryFunctionCallGeneric(
        BinaryFunctionCallGeneric::MAX,
        std::move(x),
        std::move(y));
}

} // namespace police::expressions

namespace police {

std::ostream& operator<<(std::ostream& out, const expressions::Expression& expr)
{
    expr.base()->dump(out);
    return out;
}

} // namespace police
